<?php

class QB
{

    const MYSQL_DATE_FORMAT = 'Y-m-d H:i:s';
    const TIME_UNIT_YEAR = 'year';
    const TIME_UNIT_MONTH = 'month';
    const TIME_UNIT_DAY = 'day';
    const TIME_UNIT_QUARTER = 'quarter';

    /**
     * @param SelectQuery $query
     * @param $column
     * @param $values
     * @param bool $toInt
     * @return SelectQuery
     */
    public static function whereIn(SelectQuery $query, $column, $values, $toInt = false)
    {
        return $query
            ->where(self::whereInString($column, $values, $toInt));

    }

    public static function whereInString($column, $values, $toInt = false)
    {
        $s = self::arrayToSqlString($values, $toInt);
        return "$column in ($s)";
    }

    /**
     * @param SelectQuery $query
     * @param $column
     * @param $values
     * @param bool $toInt
     * @return SelectQuery
     */
    public static function whereNotIn(SelectQuery $query, $column, $values, $toInt = false)
    {
        $s = self::arrayToSqlString($values, $toInt);
        return $query
            ->where("$column not in ($s)");

    }

    public static function whereDateFrom(SelectQuery $query, $column, DateTime $dateFrom = null, $includeFrom = true)
    {
        $symbolFrom = $includeFrom ? '>=' : '>';
        return $query->where("$column $symbolFrom ?",$dateFrom->format(self::MYSQL_DATE_FORMAT));
    }

    public static function whereDateTo(SelectQuery $query, $column, DateTime $dateTo = null, $includeTo = false)
    {
        $symbolTo = $includeTo ? '<=' : '<';
        return $query->where("$column $symbolTo ?",$dateTo->format(self::MYSQL_DATE_FORMAT));
    }


    /**
     * @param SelectQuery $query
     * @param string $column
     * @param DateTime $dateFrom
     * @param DateTime $dateTo
     * @param bool $includeFrom
     * @param bool $includeTo
     * @return SelectQuery
     */
    public static function whereDateFromTo(SelectQuery $query, $column, DateTime $dateFrom, DateTime $dateTo, $includeFrom = true, $includeTo = false)
    {
        $symbolFrom = $includeFrom ? '>=' : '>';
        $symbolTo = $includeTo ? '<=' : '<';
        $query->where("($column $symbolFrom ? AND $column $symbolTo ?)",
            $dateFrom->format(self::MYSQL_DATE_FORMAT),
            $dateTo->format(self::MYSQL_DATE_FORMAT)
        );

        return $query;
    }

    /**
     * @param SelectQuery $query
     * @param $column
     * @param $timeUnit
     * @return SelectQuery
     */
    public static function buildTimeUnit(
        SelectQuery $query,
        $column,
        $timeUnit,
        $fromTimezone = 'UTC',
        $toTimezone = 'Europe/Amsterdam'
    )
    {
        $sqlDateParam = "CONVERT_TZ($column,'$fromTimezone','$toTimezone')";
        $query->select("YEAR($sqlDateParam) `year`");
        $query->groupBy('`year`');
        $query->orderBy('`year`');
        
        if ($timeUnit === self::TIME_UNIT_QUARTER) {
            $query->select("QUARTER($sqlDateParam) `quarter`");
            $query->groupBy('`quarter`');
            $query->orderBy('`quarter`');
        }

        if ($timeUnit === self::TIME_UNIT_MONTH || $timeUnit === self::TIME_UNIT_DAY) {
            $query->select("MONTH($sqlDateParam) `month`");
            $query->groupBy('`month`');
            $query->orderBy('`month`');
        }
        
        if ($timeUnit === self::TIME_UNIT_DAY) {
            $query->select("DAY($sqlDateParam) `day`");
            $query->groupBy('`day`');
            $query->orderBy('`day`');    
        }
        
        return $query;
    }

    /**
     * @param BaseQuery $query
     * @return string
     */
    public static function getRawSql($query)
    {
        return self::interpolateQuery($query->getQuery(false), $query->getParameters());
    }

    protected static function interpolateQuery($query, $params) {
        $keys = array();

        # build a regular expression for each parameter
        foreach ($params as $key => &$value) {
            $value = is_string($value) ? "\"$value\"" : $value;
            $value = $value instanceof DateTime ? "\"{$value->format(self::MYSQL_DATE_FORMAT)}\"" : $value;
            if (is_string($key)) {
                $keys[] = '/:'.$key.'/';
            } else {
                $keys[] = '/[?]/';
            }
        }

        $query = preg_replace($keys, $params, $query, 1, $count);

        #trigger_error('replaced '.$count.' keys');

        return $query;
    }



    /**
     * @param array $a
     * @param bool $toInt
     *
     * @return string
     */
    protected static function arrayToSqlString($a, $toInt)
    {
        $s = '';
        foreach ($a as $key => $val) {
            if ($toInt) {
                $s .= (int) $val;
            } else {
                $s .= "'$val'";
            }

            if ($key < count($a) - 1) {
                $s .= ',';
            }
        }

        return $s;
    }
}