<?php

namespace Senh\Lib\Managers;
use Senh\Lib\Models\InternalProductModel;
use Senh\Lib\Models\InternalProductPropertiesModel;

class InternalProductManager
{

    const VORM_FYSIEK = 'fysiek';
    const VORM_DIGITAAL = 'digitaal';

    /**
     * @var InternalProductModel[]
     */
    protected static $internalProductModels;

    /**
     * Call this method to get singleton
     * @deprecated: use di container
     * @return InternalProductManager
     */
    public static function getInstance()
    {
        static $instance = null;
        if ($instance === null) {
            $instance = new self();
        }

        return $instance;
    }

    public static function setInternalProductModels($internalProductModels)
    {
        self::$internalProductModels = $internalProductModels;
    }

    /**
     * @return InternalProductModel[]
     */
    public function getAll()
    {
        return self::$internalProductModels;
    }

    /**
     * @param $productId
     *
     * @return InternalProductModel|null
     */
    public function getSingle($productId)
    {
        foreach ($this->getAll() as $product) {
            if ($product->getProductId() == $productId) {
                return $product;
            }
        }

        return null;
    }

    /**
     * @param $productId
     *
     * @return bool
     */
    public function exists($productId)
    {
        return $this->getSingle($productId) !== null;
    }

    /**
     * @deprecated, use ::isPhysical1 instead
     * @param $productId
     * @return bool
     */
    public function isFysiek($productId)
    {
        return $this->isPhysical1($productId);
    }

    public function isPhysical1($productId)
    {
        return in_array($productId, $this->getProductIdsPhysical1());
    }

    /**
     * @deprecated, use ::getProductIdsPhysical1 instead
     * @return array
     */
    public function getProductIdsFysiek()
    {
        return $this->getProductIdsPhysical1();
    }

    /**
     * @return array
     */
    public function getProductIdsPhysical1()
    {
        $fysiek = array(
            44685, // fysiek Lekker Zwanger Worden
            46948, // fysiek Smoothie Afslankbijbel
            23616, // fysiek Gespierde God
            42089, // fysiek Alles over Groene Smoothies
            21142, // arb
            37542, // sbb
        );

        foreach ($this->getAll() as $internalProductModel) {
            if ($internalProductModel->getProductType1() === InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL) {
                $fysiek[] = $internalProductModel->getProductId();
            }
        }

        return array_unique($fysiek);
    }

    /**
     * @return int[]
     * @throws \Exception
     */
    public function getProductIdsVatLow1VatHighN()
    {
        $productIds = [];
        foreach ($this->getAll() as $internalProductModel) {
            if (!$internalProductModel->hasPropertiesN()) {
                continue;
            }
            $vat1 = \VatHelper::getVatPercentageByCategoryAndType(
                $internalProductModel->getProductCategory1(),
                $internalProductModel->getProductType1());
            $vatN = \VatHelper::getVatPercentageByCategoryAndType(
                $internalProductModel->getProductCategoryN(),
                $internalProductModel->getProductTypeN());

            if (\VatHelper::isVatLow($vat1) && \VatHelper::isVatHigh($vatN)) {
                $productIds[] = $internalProductModel->getProductId();
            }
        }

        return $productIds;
    }

    /**
     * @deprecated, this method is unused and the returned data is incomplete!
     * @return array
     */
    public function getProductIds()
    {
        return array(
            49151,
            47094,
            46948,
            46945,
            44685,
            42089,
            38678,
            38520,
            37542,
            37464,
            36881,
            36622,
            34196,
            34100,
            33870,
            33869,
            33867,
            32978,
            31562,
            31384,
            27646,
            25164,
            25007,
            23616,
            22505,
            21907,
            21631,
            21142,
            20406,
            20174,
            // stopped at page 5

        );
    }
    
    public function getAllProductCountryCodes()
    {
        $countries = [];
        foreach ($this->getAll() as $model)
        {
            if (!in_array($model->getProductCountryCode(), $countries)) {
                $countries[] = strtolower($model->getProductCountryCode());
            }
        }

        return $countries;
    }

    /**
     * @return string[]
     */
    public function getAvailableProductCategories()
    {
        return [
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_CATEGORY_FOOD,
            InternalProductPropertiesModel::PRODUCT_CATEGORY_PROGRAM,
        ];
    }

    /**
     * @return string[]
     */
    public function getAvailableProductTypes()
    {
        return [
            InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
        ];
    }

    /**
     * Very important note, HUAP product ids that are not defined in the InternalProductModels are not included and must
     * be fetched from the sales database.
     * HUAP product ids have categoryN 'PRODUCT_CATEGORY_PROGRAM' and typeN 'PRODUCT_TYPE_DIGITAL'
     * @param string $categoryN
     * @param string $typeN
     * @return array
     */
    public function getProductIdsByCategoryAndTypeN(
        $categoryN,
        $typeN
    ) {
        $productIds = [];
        foreach ($this->getAll() as $model) {
            if (
                $model->getProductCategoryN() === $categoryN &&
                $model->getProductTypeN() === $typeN
            ) {
                $productIds[] = $model->getProductId();
            }
        }

        return $productIds;
    }

    /**
     * @param \SequenceModel $sequence
     * @return string
     */
    public function getProductTypeBySequence($sequence)
    {
        $internalProduct = $this->getSingle($sequence->getSale()->getProductId());
        if (!$internalProduct) {
            return $sequence->getSequenceNumber() > 1 || !$this->isPhysical1($sequence->getSale()->getProductId())
                ? InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL
                : InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL;
        }

        return $sequence->getSequenceNumber() > 1 ? $internalProduct->getProductTypeN() : $internalProduct->getProductType1();
    }

}