<?php

namespace Senh\Errors;

use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Handler\SwiftMailerHandler;
use Monolog\Logger;

class ErrorHandler
{
    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var SwiftMailerHandler
     */
    protected $mailHandler;

    /**
     * No trailing slash
     * @var string | null
     */
    protected $logDir;

    /**
     * Call this method to get singleton
     *
     * @return ErrorHandler
     */
    public static function getInstance($logDir = null)
    {
        static $instance = null;
        if ($instance === null) {
            $logDir = !$logDir && defined('LOG_DIR') ? LOG_DIR : $logDir;
            $instance = new self($logDir);
        }

        return $instance;
    }

    protected function __construct($logDir)
    {
        $this->logDir = $logDir;
        $mailer = \MailerFactory::getInstance()->getMailer();
        $this->mailHandler = new FingersCrossedHandler(
            new SwiftMailerHandler($mailer,array($this, 'createMessage'), Logger::INFO),
            Logger::ALERT,
            10000,
            true,
            false,
            Logger::ERROR
        );

        $date = new \DateTime('now', new \DateTimeZone('Europe/Amsterdam'));
        $fileName = $this->logDir . '/error-handler/main--'.$date->format('Y-m-d').'.log';
        $streamHandler = new FingersCrossedHandler(
            new StreamHandler($fileName, Logger::INFO, true, 0777),
            Logger::ERROR,
            10000,
            true,
            false,
            null
        );

        $this->logger = new Logger('error-handler');
        $this->logger->pushHandler($this->mailHandler);
        $this->logger->pushHandler($streamHandler);
    }

    public function createMessage()
    {
        $extraSubject = ', source: ' . php_sapi_name();
        $message = new \Swift_Message('Server error'.$extraSubject, 'aap', 'text/plain');
        $message->setFrom('info@senhuitgeverij.nl');
        $message->setTo(explode(',',EMAIL_TECHGUYS));

        return $message;
    }

    public function handleErrors()
    {
        $errorHandler = self::getInstance();
        set_error_handler(array($errorHandler, 'onError'), E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR | E_USER_ERROR);
//        set_error_handler(array($errorHandler, 'onError'), E_ALL);
        set_exception_handler(array($errorHandler, 'onException'));
        register_shutdown_function(array($errorHandler, 'onShutdown'));
    }

    /**
     * @param int $errno: the level of the error raised
     * @param string $errstr: the error message
     * @param string $errfile
     * @param int $errline
     * @param array $errcontext
     */
    public function onError($errno , $errstr, $errfile = null, $errline = null, $errcontext = null)
    {
        switch($errno) {
            case E_ERROR:
            case E_PARSE:
            case E_CORE_ERROR:
            case E_COMPILE_ERROR:
            case E_USER_ERROR:
                $context = array(
                    'message' => $errstr,
                    'code' => $errno,
                    'file' => $errfile,
                    'line' => $errline,
//                    'context' => $errcontext, // seems to break normalisation in vendor/monolog/monolog/src/Monolog/Formatter/NormalizerFormatter.php
                );
                if ($errcontext && isset($errcontext['_SERVER'])) {
                    $context['server'] = $errcontext['_SERVER'];
                }

                $this->logger->addCritical('Error', $context);
                $this->showUserErrorMessage();
        }
    }

    /**
     * @param \Exception $exception
     */
    public function onException($exception)
    {
        $context = array(
            'message' => $exception->getMessage(),
            'code' => $exception->getCode(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'previous' => $exception->getPrevious(),
            'trace' => $exception->getTrace(),
        ) ;

        $exception->getTrace();
        $this->logger->addCritical('Exception', $context);

        if (ENV === ENV_DEV) {
            echo '<pre>';
            print_r($context);
            echo '</pre>';
        } else {
            $this->showUserErrorMessage();
        }
    }

    public function onShutdown()
    {
        if ($error = error_get_last()){
            $this->onError($error['type'], $error['message'], $error['file'], $error['line']);
        }
    }

    protected function showUserErrorMessage()
    {
        echo <<<EOD
<div style="text-align: center">
    <h1>Onze excuses, we hebben een technisch probleem.</h1>
    <p>We zijn inmiddels op de hoogte en gaan zo snel mogelijk voor je aan de slag om het op te lossen.</p>
</div>
EOD;
    }

    public function logRequest()
    {
        if (isset($_SERVER)) {
            $this->logger->info('request', $_SERVER);
        }
    }

    /**
     * @return SwiftMailerHandler
     */
    public function getMailHandler()
    {
        return $this->mailHandler;
    }

    /**
     * @return Logger
     */
    public function getLogger()
    {
        return $this->logger;
    }

    /**
     * @param Logger $logger
     */
    public function setLogger($logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return null|string
     */
    public function getLogDir()
    {
        return $this->logDir;
    }
}