<?php

class DateHelper
{
    const DATE_TIME_FORMAT_MYSQL = 'Y-m-d H:i:s';
    const DATE_FORMAT_MYSQL = 'Y-m-d';
    const DATE_TIME_FORMAT_HUMAN_ADMINSTATIE = 'd-m-Y H:i:s';
    const DATE_TIME_FORMAT_API = 'Y-m-d H:i:s';
    const DATE_TIME_FORMAT_PAYPRO_TRANSACTIONS = 'm/d/Y H:i';
    const DATE_TIME_FORMAT_PAYPRO = 'd/m/Y H:i';
    const DATE_TIME_FORMAT_DUTCH = 'd-m-Y';

    /**
     * @param string $format
     * @param string $date
     * @param string | null $fromTimeZone
     * @param string | null $toTimeZone
     * @return null | DateTime
     */
    public static function toDateTime($format, $date, $fromTimeZone='UTC', $toTimeZone='UTC')
    {
        if (!$date) {
            return null;
        }

        $fromTimeZone = $fromTimeZone ?: 'UTC';
        $dateTime = DateTime::createFromFormat($format, $date, new DateTimeZone($fromTimeZone));
        if ($toTimeZone) {
            $dateTime->setTimezone(new DateTimeZone($toTimeZone));
        }

        return $dateTime;
    }

    /**
     * @param DateTime $date
     * @param string $format
     * @param string $toTimeZone
     * @return string
     */
    public static function toFormat($date, $format, $toTimeZone='UTC')
    {
        if (!$date instanceof DateTime) {
            return '';
        }

        if ($toTimeZone && $date->getTimezone()->getName() !== $toTimeZone) {
            $date = clone $date;
            $date->setTimezone(new DateTimeZone($toTimeZone));
        }

        return $date->format($format);
    }

    /**
     * @param DateTime $date
     * @return string
     */
    public static function toSqlFormat(DateTime $date, $toTimeZone='UTC')
    {
        return self::toFormat($date, self::DATE_TIME_FORMAT_MYSQL, $toTimeZone);
    }

    /**
     * @param string $date
     * @return DateTime|null
     */
    public static function fromSqlFormat($date, $fromTimeZone='UTC', $toTimeZone='UTC')
    {
        return self::toDateTime(self::DATE_TIME_FORMAT_MYSQL, $date, $fromTimeZone, $toTimeZone);
    }

    /**
     * @param string $date
     * @return DateTime|null
     */
    public static function fromPayproFormat($date, $fromTimeZone='Europe/Amsterdam', $toTimeZone='UTC')
    {
        return self::toDateTime(self::DATE_TIME_FORMAT_PAYPRO, $date, $fromTimeZone, $toTimeZone);
    }

    /**
     * @param DateTime $date
     * @return string
     */
    public static function toPayproTransactionFormat($date, $toTimeZone='Europe/Amsterdam')
    {
        return self::toFormat($date, self::DATE_TIME_FORMAT_PAYPRO_TRANSACTIONS, $toTimeZone);
    }

    /**
     * @param string $date
     * @return DateTime|null
     */
    public static function fromPayProTransactionFormat($date)
    {
        return DateHelper::toDateTime(self::DATE_TIME_FORMAT_PAYPRO_TRANSACTIONS, $date);
    }

    /**
     * @param string $date
     * @return DateTime|null
     */
    public static function fromDutchFormat($date, $fromTimeZone='Europe/Amsterdam', $toTimeZone=null)
    {
        return DateHelper::toDateTime(self::DATE_TIME_FORMAT_DUTCH, $date, $fromTimeZone, $toTimeZone);
    }

    /**
     * @param DateTime $date
     * @return string
     */
    public static function toHumanAdministratieFormat(DateTime $date, $toTimeZone='Europe/Amsterdam')
    {
        return self::toFormat($date, self::DATE_TIME_FORMAT_HUMAN_ADMINSTATIE, $toTimeZone);
    }

    /**
     * @param $date
     * @param string $fromTimeZone
     * @param string $toTimeZone
     * @return \DateTime
     */
    public static function fromApiDateFormat($date, $fromTimeZone = 'Europe/Amsterdam', $toTimeZone = 'UTC')
    {
        return DateHelper::toDateTime(self::DATE_TIME_FORMAT_API, $date . ' 00:00:00', $fromTimeZone, $toTimeZone);
    }

    /**
     * @param int $year
     * @param int $quarter
     * @param string $fromTimeZone
     * @param string $toTimeZone
     * @return DateTime|null
     */
    public static function toYearQuarterFromDate($year, $quarter, $fromTimeZone='Europe/Amsterdam', $toTimeZone='UTC')
    {
        $month = ($quarter - 1) * 3 + 1;
        $day = 1;

        return self::toDateTime('Y-m-d H:i:s', "$year-$month-$day 00:00:00", $fromTimeZone, $toTimeZone);
    }

    /**
     * @param int $year
     * @param int $quarter
     * @param string $fromTimeZone
     * @param string $toTimeZone
     * @return DateTime|null
     */
    public static function toYearQuarterToDate($year, $quarter, $fromTimeZone='Europe/Amsterdam', $toTimeZone='UTC')
    {
        $month = ($quarter) * 3 + 1;
        $day = 1;

        return self::toDateTime('Y-m-d H:i:s', "$year-$month-$day 00:00:00", $fromTimeZone, $toTimeZone);
    }

    /**
     * @param int $year
     * @param int $month
     * @param string $fromTimeZone
     * @param string $toTimeZone
     * @return DateTime|null
     */
    public static function toYearMonthFromDate($year, $month, $fromTimeZone='Europe/Amsterdam', $toTimeZone='UTC')
    {
        $day = 1;
        return self::toDateTime('Y-m-d H:i:s', "$year-$month-$day 00:00:00", $fromTimeZone, $toTimeZone);
    }

    /**
     * @param int $year
     * @param int $month
     * @param string $fromTimeZone
     * @param string $toTimeZone
     * @return DateTime|null
     */
    public static function toYearMonthToDate($year, $month, $fromTimeZone='Europe/Amsterdam', $toTimeZone='UTC')
    {
        $month++;
        $day = 1;
        return self::toDateTime('Y-m-d H:i:s', "$year-$month-$day 00:00:00", $fromTimeZone, $toTimeZone);
    }


    /**
     * @param string $date
     * @return DateTime
     */
    public static function fromIsoDateFormat($date, $toTimeZone='UTC')
    {
        $date = DateTime::createFromFormat(DateTime::ATOM, $date);
        $date->setTimezone(new DateTimeZone($toTimeZone));

        return $date;
    }

    /**
     * @param DateTime $date
     * @param string $toTimeZone
     * @return string
     */
    public static function toIsoDateFormat($date, $toTimeZone = null)
    {
        return self::toFormat($date, DateTime::ATOM, $toTimeZone);
    }

    /**
     * @param DateTime $date
     * @param int $hour
     * @param int $minute
     * @param int $second
     * @param int $microseconds
     * @return DateTime
     */
    public static function setTime($date, $hour=0, $minute=0, $second=0, $microseconds=0)
    {
        return new DateTime($date->format(self::DATE_FORMAT_MYSQL)." $hour:$minute:$second.$microseconds", $date->getTimezone());
    }
}
