<?php

abstract class CoreManager
{
    /**
     * @var PDO
     */
    protected $pdo;

    /**
     * @var FluentPDOSenh
     */
    protected $fpdo;

    /**
     * @var ModelInterface
     */
    protected $lastAffectedModel;

    /**
     * @var string
     */
    protected $debugQuery;

    public static function getInstance()
    {
        static $instance = null;
        if ($instance === null) {
            $instance = new static();
        }

        return $instance;
    }

    public function __construct()
    {
        $this->pdo = DbHandler::getInstance()->getPdo();
        $this->fpdo = DbHandler::getInstance()->getFpdo();
    }

    /**
     * @return int|null
     */
    public function getLastAffectedId()
    {
        if ($this->lastAffectedModel instanceof ModelInterface) {
            return $this->lastAffectedModel->getId();
        }

        return null;
    }

    /**
     * @return ModelInterface
     */
    public function getLastAffectedModel()
    {
        return $this->lastAffectedModel;
    }

    /**
     * Overwrite by children
     * @return string
     */
    abstract protected function getTableName();

    /**
     * @return array
     */
    abstract protected function getDbProperties();

    /**
     * @param $dbData
     * @return mixed
     */
    abstract protected function createNew($dbData);

    protected function getQuotedTableName()
    {
        return "`{$this->getTableName()}`";
    }


    /**
     * @return string
     */
    protected function getDbFieldPrefix()
    {
        return $this->getTableAlias().'____';
    }

    /**
     * Overwrite by children
     * @return string
     */
    protected function getTableAlias()
    {
        return $this->getTableName();
    }

    /**
     * @return array
     */
    protected function getDbFields()
    {
        $dbFields = array();
        foreach ($this->getDbProperties() as $dbProperty) {
            $dbFields[] = StringHelper::camelCasedToUnderScored($dbProperty);
        }

        return $dbFields;
    }

    public function truncate()
    {
        $this->pdo
            ->query('TRUNCATE '.$this->getQuotedTableName())
            ->execute();
    }

    /**
     * @param string $dbField
     * @return string
     */
    public function toPrefixedDbField($dbField)
    {
        return $this->getDbFieldPrefix().$dbField;
    }

    /**
     * @param array $dbFields
     * @return array: dbField => prefixedDbField
     */
    public function toPrefixedDbFields($dbFields)
    {
        $pfDbFields = array();
        foreach ($dbFields as $dbField) {
            $pfDbFields[$dbField] = $this->toPrefixedDbField($dbField);
        }

        return $pfDbFields;
    }

    /**
     * @return array
     */
    public function getPrefixedDbFields()
    {
        return $this->toPrefixedDbFields($this->getDbFields());
    }

    /**
     * @param string $dbField
     * @return string
     */
    public function removePrefix($dbField)
    {
        if (strpos($dbField, $this->getDbFieldPrefix()) === 0) {
            return substr($dbField, strlen($this->getDbFieldPrefix()));
        }

        return $dbField;
    }

    /**
     * @param $dbData
     * @param $dbField
     * @return mixed
     */
    protected function getDbValue($dbData, $dbField)
    {
        return isset($dbData[$this->toPrefixedDbField($dbField)]) ? $dbData[$this->toPrefixedDbField($dbField)] : $dbData[$dbField];
    }

    /**
     * @param ModelInterface $model
     * @return bool
     */
    public function create($model)
    {
        $now = new DateTime('now', $this->getModelTimeZone());
        if (method_exists($model, 'setCreatedAt')) {
            $model->setCreatedAt($now);
        }
        if (method_exists($model, 'setUpdatedAt')) {
            $model->setUpdatedAt($now);
        }
        $data = $this->fromModelToDbData($model);
        $query = $this->fpdo->insertInto($this->getQuotedTableName(), $data);
        $success = $query->execute() !== false;
        if ($success) {
            $this->lastAffectedModel = $model;
            if (!$model->getId()) {
                $model->setId((int)$this->fpdo->getPdo()->lastInsertId());
            }
        } else {
            $this->debugQuery = QB::getRawSql($query);
        }

        return $success;
    }

    /**
     * @param ModelInterface $model
     * @return bool
     */
    public function update($model)
    {
        $now = new DateTime('now', $this->getModelTimeZone());
        if (method_exists($model, 'setUpdatedAt')) {
            $model->setUpdatedAt($now);
        }
        $data = $this->fromModelToDbData($model);
        unset($data['id']);
        if (array_key_exists('created_at', $data)) {
            unset($data['created_at']);
        }

        $query = $this->fpdo->update($this->getQuotedTableName(), $data)
            ->where('id', $model->getId());
        $success = $query->execute() !== false;
        if ($success) {
            $this->lastAffectedModel = $model;
        } else {
            $this->debugQuery = QB::getRawSql($query);
        }

        return $success;
    }

    /**
     * @param ModelInterface $model
     * @return bool
     */
    public function deleteModel($model)
    {
        $query = $this->getFpdo()->delete($this->getQuotedTableName());
        $query->where('id', $model->getId());

        $success = $query->execute() !== false;
        if (!$success) {
            $this->debugQuery = QB::getRawSql($query);
        }

        return $success;
    }

    /**
     * @param array $criteria
     * @return bool
     */
    public function delete($criteria)
    {
        if (!$criteria) {
            return null;
        }

        $dbTimeZone = $this->getDbTimeZone();
        $toSqlValue = function ($value) use (&$toSqlValue, $dbTimeZone) {
            if (is_array($value)) {
                foreach ($value as &$val) {
                    $val = $toSqlValue($val);
                }
            }
            if ($value instanceof DateTime) {
                $value = DateHelper::toSqlFormat($value, $dbTimeZone);
            }

            return $value;
        };

        $query = $this->getFpdo()->delete($this->getQuotedTableName());
        $query->disableSmartJoin();
        foreach ($criteria as $column => $values) {
            $values = $toSqlValue($values);
            $query->where("{$this->getTableName()}.$column", $values);
        }

        $success = $query->execute() !== false;
        if (!$success) {
            $this->debugQuery = QB::getRawSql($query);
        }

        return $success;
    }

    /**
     * @param ModelInterface $model
     * @return ModelInterface|null
     */
    public function fetchFreshModel($model)
    {
        return $this->getSingleModel(array('id' => $model->getId()));
    }

    /**
     * @param ModelInterface: $model
     * @return array
     */
    public function fromModelToDbData($model)
    {
        $dbProperties = $this->getDbProperties();
        $dbData = array();
        foreach ($dbProperties as $dbProperty) {
            $dbField = StringHelper::camelCasedToUnderScored($dbProperty);
            $method = 'get'.ucfirst($dbProperty);
            $value = $model->$method();
            if (is_bool($value)) {
                $value = (int)$value;
            }
            if ($value instanceof DateTime) {
                $value = clone $value;
                $value->setTimezone($this->getDbTimeZone());
                $value = $value->format(QB::MYSQL_DATE_FORMAT);
            }
            if (is_array($value)) {
                $value = json_encode($value);
            }
            if (null !== $value) {
                $dbData[$dbField] = $value;
            }
        }

        return $dbData;
    }

    /**
     * @param iterable $dbData
     * @return array
     */
    public function fromDbDataToModels($dbData)
    {
        $models = array();
        foreach ($dbData as $row) {
            $model = $this->fromDbDataToModel($row);
            $models[] = $model;
        }

        return $models;
    }

    /**
     * @param array $dbData
     * @return object|mixed
     */
    public function fromDbDataToModel($dbData)
    {
        $dbProperties = $this->getDbProperties();
        $modelData = array();
        foreach ($dbProperties as $dbProperty) {
            $dbField = StringHelper::camelCasedToUnderScored($dbProperty);
            $value = $this->getDbValue($dbData, $dbField);
            $dateRegEx = '/(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/';
            preg_match($dateRegEx, $value, $result);
            if ($result) {
                $value = DateTime::createFromFormat(QB::MYSQL_DATE_FORMAT, $value, $this->getDbTimeZone());
                $value->setTimezone($this->getModelTimeZone());
            }

            $modelData[$dbField] = $value;
        }
        $model = $this->createNew($modelData);

        return $model;
    }

    /**
     * @param array | string $fields
     * @return SelectQuery|DeleteQuery|SelectQuerySenh
     */
    protected function buildSelectQueryStart($fields = '*')
    {
        $query = $this->fpdo->from($this->getTableName());
        if ($fields === '*') {
            $fields = $this->getDbFields();
        }

        $query->select(null);
        $fields = $fields === null ? array() : $fields;
        $fields = is_array($fields) ? $fields : array($fields);
        $fields = $this->toPrefixedDbFields($fields);
        $query = self::buildSelectPrefixedFields($query, $this, $fields);

        return $query;
    }


    protected static function buildSelectPrefixedFields($query, BaseManager $manager, $pfFields)
    {
        foreach ($pfFields as $field => $pfField) {
            $query->select("`{$manager->getTableAlias()}`.`$field` as $pfField");
        }

        return $query;
    }

    /**
     * @param array $criteria
     * @return array | bool
     */
    protected function getRows($criteria = null, $limit = null)
    {
        $dbTimeZone = $this->getDbTimeZone();
        $toSqlValue = function ($value) use (&$toSqlValue, $dbTimeZone) {
            if (is_array($value)) {
                foreach ($value as &$val) {
                    $val = $toSqlValue($val);
                }
            }
            if ($value instanceof DateTime) {
                $value = DateHelper::toSqlFormat($value, $dbTimeZone);
            }

            return $value;
        };

        $query = $this->buildSelectQueryStart();
        if ($criteria) {
            foreach ($criteria as $column => $values) {
                $values = $toSqlValue($values);
                $query->where("{$this->getTableName()}.$column", $values);
            }
        }

        if ($limit) {
            $query->limit($limit);
        }

        return $query->fetchAll();
    }

    /**
     * @param array $criteria
     * @return ModelInterface[]
     */
    public function getModels($criteria = null, $limit = null)
    {
        $rows = $this->getRows($criteria, $limit);
        if (!$rows) {
            return array();
        }
        $models = array();
        foreach ($rows as $row) {
            $models[] = $this->fromDbDataToModel($row);
        }

        return $models;
    }

    /**
     * @param array $criteria
     * @return ModelInterface | null
     */
    public function getSingleModel($criteria = null)
    {
        $models = $this->getModels($criteria, 1);
        if (!count($models)) {
            return null;
        }

        return reset($models);
    }

    /**
     * @deprecated use getSingleModel instead
     * @param string|int $value
     * @param string $identifier
     * @return Mixed|null
     */
    public function getSingleModelOld($value, $identifier = 'id')
    {
        return $this->getSingleModel(array($identifier => $value));
    }

    /**
     * @param array $criteria
     * @return int
     */
    public function getCount(array $criteria = null)
    {
        $query = $this->buildSelectQueryStart()
            ->select(null)
            ->select("count(*)");

        if ($criteria) {
            foreach ($criteria as $column => $value) {
                $query->where($column, $value);
            }
        }

        return (int)$query->fetchColumn();
    }

    protected static function buildJoinHelper(
        $query,
        $joinType,
        $baseTableOrAlias,
        $foreignTable,
        $foreignAlias = null,
        $joinColumn = null,
        $foreignColumn = null,
        $extraStm = null
    ) {
        $joinColumn = $joinColumn ?: "{$foreignTable}_id";
        $foreignColumn = $foreignColumn ?: 'id';
        $foreignTableOrAlias = $foreignAlias ?: $foreignTable;
        $stm = "`$foreignTable` $foreignAlias ON `$baseTableOrAlias`.`$joinColumn` = `$foreignTableOrAlias`.`$foreignColumn` $extraStm";

        return self::buildJoinTypeHelper($query, $stm, $joinType);
    }

    /**
     * @param SelectQuery|DeleteQuery | DeleteQuery $query
     * @param string $stm
     * @param string $joinType
     * @return SelectQuery|DeleteQuery
     */
    protected static function buildJoinTypeHelper($query, $stm, $joinType = 'left')
    {
        if ($joinType === 'left') {
            $query->leftJoin($stm);
        }
        if ($joinType === 'inner') {
            $query->innerJoin($stm);
        }

        return $query;
    }

    public static function getDbTimeZone()
    {
        return new DateTimeZone('UTC');
    }

    public static function getModelTimeZone()
    {
        return new DateTimeZone('UTC');
    }

    /**
     * @param DateTime $date
     * @param bool $clone
     * @return DateTime
     */
    public static function toDbDate($date, $clone = true)
    {
        $date = $clone ? clone $date : $date;
        $date->setTimezone(self::getDbTimeZone());

        return $date;
    }

    /**
     * @param DateTime $date
     * @param bool $clone
     * @return DateTime
     */
    public static function toModelDate($date, $clone = true)
    {
        $date = $clone ? clone $date : $date;
        $date->setTimezone(self::getModelTimeZone());

        return $date;
    }

    /**
     * @return PDO
     */
    public function getPdo()
    {
        return $this->pdo;
    }

    /**
     * @return FluentPDOSenh
     */
    public function getFpdo()
    {
        return $this->fpdo;
    }

    /**
     * @return string
     */
    public function getDebugQuery()
    {
        return $this->debugQuery;
    }
}