<?php

namespace Senh\Lib\NextSell;

use ConsumerModel;
use SaleModel;

class NextSellHelper
{
    const SESSION_CHECKOUT_INTERNAL_SALE_ID = 'NEXT_SELL_CHECKOUT_SALE_ID';
    const SESSION_CHECKOUT_SALE = 'NEXT_SELL_CHECKOUT_SALE';
    const SESSION_CHECKOUT_PAYMENT_METHOD = 'NEXT_SELL_CHECKOUT_PAYMENT_METHOD';
    const SESSION_CHECKOUT_CONSUMER = 'NEXT_SELL_CHECKOUT_CONSUMER';
    const SESSION_PAYMENT_HASH = 'NEXT_SELL_PAYMENT_HASH';
    const SESSION_FETCHED_SALE = 'NEXT_SELL_FETCHED_SALE';
    const SESSION_SUCCESS = 'NEXT_SELL_SUCCESS';
    const SESSION_FREEZE = 'NEXT_SELL_FREEZE';

    /**
     * @var SaleModel
     */
    protected $sale;

    /**
     * @var int
     */
    protected $checkoutInternalSaleId;

    /**
     * @var string
     */
    protected $paymentHash;

    /**
     * @var array: enriched checkout data
     */
    protected $checkoutSale;

    /**
     * @var ConsumerModel
     */
    protected $checkoutConsumer;

    /**
     * @var string
     */
    protected $checkoutPaymentMethod;

    /**
     * @var bool
     */
    protected $success;

    /**
     * @var bool
     */
    protected $freeze;

    public function __construct()
    {
        if (session_id() == '') {
            session_start();
        }

        $_SESSION[self::SESSION_SUCCESS] = isset($_SESSION[self::SESSION_SUCCESS]) ? $_SESSION[self::SESSION_SUCCESS] : false;
        $_SESSION[self::SESSION_FREEZE] = isset($_SESSION[self::SESSION_FREEZE]) ? $_SESSION[self::SESSION_FREEZE] : false;

        if ($this->hasValidSessionData()) {
            $this->setPropertiesFromSession();
        }
    }

    /**
     * @param $checkoutSale: the sale array
     * @param ConsumerModel $consumer
     */
    public function saveCheckoutData($internalSaleId, $checkoutSale, $consumer, $paymentMethod, $paymentHash)
    {
        $_SESSION[self::SESSION_CHECKOUT_INTERNAL_SALE_ID] = $internalSaleId;
        $_SESSION[self::SESSION_CHECKOUT_SALE] = $checkoutSale;
        $_SESSION[self::SESSION_CHECKOUT_CONSUMER] = serialize($consumer);
        $_SESSION[self::SESSION_CHECKOUT_PAYMENT_METHOD] = $paymentMethod;
        $_SESSION[self::SESSION_PAYMENT_HASH] = $paymentHash;
        $_SESSION[self::SESSION_SUCCESS] = false;
        $_SESSION[self::SESSION_FREEZE] = false;
    }

    /**
     * @return int
     */
    public function getCheckoutInternalSaleId()
    {
        return $this->checkoutInternalSaleId;
    }

    /**
     * @return array
     */
    public function getCheckoutSale()
    {
        return $this->checkoutSale;
    }

    /**
     * @return ConsumerModel
     */
    public function getCheckoutConsumer()
    {
        return $this->checkoutConsumer;
    }

    /**
     * @return string
     */
    public function getPaymentHash()
    {
        return $this->paymentHash;
    }

    /**
     * @return SaleModel
     */
    public function getSale()
    {
        return $this->sale;
    }

    public function hasSale()
    {
        return $this->sale !== null;
    }

    /**
     * @return $this
     */
    public function fetchSale($forceRefresh = false)
    {
        if (!$forceRefresh && isset($_SESSION[self::SESSION_FETCHED_SALE])) {
            $this->sale = new SaleModel($_SESSION[self::SESSION_FETCHED_SALE]);
        } else {
            $this->sale = senh_get_sale_by_hash($this->paymentHash);
            if ($this->sale !== null) {
                $_SESSION[self::SESSION_FETCHED_SALE] = $this->sale->getOriginalData();
            }
        }

        if (
            $this->sale !== null &&
            defined('PAYMENT_TEST_MODE') && PAYMENT_TEST_MODE &&
            defined('PAYMENT_TEST_IBAN') && PAYMENT_TEST_IBAN
        ) {
            $this->sale->setIban('testiban');
        }

        return $this;
    }

    public function hasValidSessionData()
    {
        return isset($_SESSION[self::SESSION_CHECKOUT_INTERNAL_SALE_ID]) &&
            isset($_SESSION[self::SESSION_CHECKOUT_SALE]) &&
            isset($_SESSION[self::SESSION_CHECKOUT_CONSUMER]) &&
            isset($_SESSION[self::SESSION_CHECKOUT_PAYMENT_METHOD]) &&
            isset($_SESSION[self::SESSION_PAYMENT_HASH]) &&
            isset($_SESSION[self::SESSION_FREEZE]) &&
            isset($_SESSION[self::SESSION_SUCCESS]);
    }

    public function setPropertiesFromSession()
    {
        $this->checkoutInternalSaleId = $_SESSION[self::SESSION_CHECKOUT_INTERNAL_SALE_ID];
        $this->checkoutSale = $_SESSION[self::SESSION_CHECKOUT_SALE];
        $this->checkoutConsumer = unserialize($_SESSION[self::SESSION_CHECKOUT_CONSUMER]);
        $this->checkoutPaymentMethod = $_SESSION[self::SESSION_CHECKOUT_PAYMENT_METHOD];
        $this->paymentHash = $_SESSION[self::SESSION_PAYMENT_HASH];
        $this->success = $_SESSION[self::SESSION_SUCCESS];
        $this->freeze = $_SESSION[self::SESSION_FREEZE];
    }

    /**
     * @return bool
     */
    public function isValidNextSell($forceFetchSaleRefresh = false)
    {
        if (!$this->hasValidSessionData()) {
            return false;
        }
        $this->fetchSale($forceFetchSaleRefresh);

        if (!$this->hasSale()) {
            return false;
        };

        return $this->freeze === false;
    }

    public function getOriginalReturnUrl()
    {
        if ($this->sale->hasMultiplePayments()) {
            return $this->checkoutSale['return_url_memb'];
        }

        return $this->checkoutSale['return_url_onetime'];
    }

    public function getCheckoutPaymentMethod()
    {
        return $_SESSION[self::SESSION_CHECKOUT_PAYMENT_METHOD];
    }

    /**
     * @return bool
     */
    public function isSuccess()
    {
        return $this->success;
    }

    /**
     * @param bool $success
     */
    public function setSuccess($success)
    {
        $_SESSION[self::SESSION_SUCCESS] = $success;
        $this->success = $success;
    }

    /**
     * @return bool
     */
    public function isFreeze()
    {
        return $this->freeze;
    }

    /**
     * @param bool $freeze
     */
    public function setFreeze($freeze)
    {
        $_SESSION[self::SESSION_FREEZE] = $freeze;
        $this->freeze = $freeze;
    }
}