<?php

namespace Senh\Lib\Helpers;

use PHPUnit\Framework\TestCase;
use Senh\Lib\Models\InternalProductPropertiesModel;
use VatHelper;

class VatHelperTest extends TestCase
{
    public function testGetAvailableMossCountries()
    {
        $message = 'Date should be cloned (tested by using uncommon timezone as input)';
        $transactionDate = new \DateTime('2018-01-01', new \DateTimeZone('Africa/Abidjan'));
        VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame('Africa/Abidjan', $transactionDate->getTimezone()->getName(), $message);

        $message = 'CH + AT threshold';
        $transactionDate = new \DateTime('2018-12-31', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de', 'fr'], $actual, $message);
        $transactionDate = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de', 'fr', 'ch', 'at'], $actual, $message);

        $message = 'FR threshold';
        $transactionDate = new \DateTime('2018-08-31', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de'], $actual, $message);
        $transactionDate = new \DateTime('2018-09-01', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de', 'fr'], $actual, $message);

        $message = 'Timezone convert';
        $transactionDate = new \DateTime('2018-09-01', new \DateTimeZone('GMT+12'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de'], $actual, $message);

        $message = 'DE threshold';
        $transactionDate = new \DateTime('2018-06-30', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be'], $actual, $message);
        $transactionDate = new \DateTime('2018-07-01', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::getAvailableMossCountries($transactionDate);
        $this->assertSame(['nl', 'be', 'de'], $actual, $message);
    }

    public function testIsMossProduct()
    {
        $message = 'MOSS product';
        $actual = VatHelper::isMossProduct(InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK, InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL);
        $this->assertTrue($actual, $message);

        $message = 'No MOSS product';
        $actual = VatHelper::isMossProduct(InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK, InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL);
        $this->assertFalse($actual, $message);
    }

    public function testIsMossCountry()
    {
        $message = 'AT threshold';
        $transactionDate = new \DateTime('2018-12-31', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::isMossCountry('at', $transactionDate);
        $this->assertFalse($actual, $message);
        $transactionDate = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $actual = VatHelper::isMossCountry('at', $transactionDate);
        $this->assertTrue($actual, $message);

    }

    public function testGetVatPercentageByCategoryAndType()
    {
        $message = 'VAT low for NL physical book';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_LOW;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL,
            $date,
            'nl'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT high for NL digital book';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_HIGH;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            'nl'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT low for DE digital book';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_LOW_DE;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            'de'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT high nl for DE digital book, before 2019, after MOSS inclusion';
        $date = new \DateTime('2018-12-31', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_HIGH;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            'de'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT low nl for DE physical book';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_LOW;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL,
            $date,
            'de'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT low nl for CH physical book';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_LOW;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL,
            $date,
            'ch'
        );
        $this->assertSame($expected, $actual, $message);

        $message = 'VAT high FR for FR digital book, after 2019';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_HIGH_FR;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            'fr'
        );

        $this->assertSame($expected, $actual, $message);
        $message = 'FR UPPERCASE test';
        $date = new \DateTime('2019-01-01', new \DateTimeZone('Europe/Amsterdam'));
        $expected = VatHelper::VAT_HIGH_FR;
        $actual = VatHelper::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            'FR'
        );
        $this->assertSame($expected, $actual, $message);
    }
}
