<?php

namespace Senh\Lib\Models;

use DI\ContainerBuilder;
use Senh\Lib\Managers\InternalProductManager;
use Senh\Lib\Prices\KortingFactory;
use PHPUnit\Framework\TestCase;
use DI\Container;
use Senh\Lib\Prices\PricesHelper;
use Senh\Lib\Prices\KortingModel;

/**
 * @coversDefaultClass \Senh\Lib\Prices\PricesHelper
 */
class PricesHelperTest extends TestCase
{

    /**
     * @var Container
     */
    protected $container;

    /**
     * @var InternalProductManager
     */
    protected $internalProductManager;

    /**
     * @var KortingFactory
     */
    protected $kortingFactory;

    /**
     * @var PricesHelper
     */
    protected $pricesHelper;


    protected function setUp()
    {
        $builder = new ContainerBuilder();
        $this->container = $builder->build();

        require __DIR__.'/../config/internalProductModels.php';
        $this->internalProductManager = $this->container->get(InternalProductManager::class);
        $this->internalProductManager->setInternalProductModels($config['internalProductModels']);

        $this->kortingFactory = $this->container->get(KortingFactory::class);
        $this->pricesHelper = $this->container->get(PricesHelper::class);

        $this->pricesHelper->setMinimalPrice1(2);
        $this->pricesHelper->setMinimalPriceN(3);
    }

    protected function getKortingModelBookDigital()
    {
        return new KortingModel('a', 9999, 1, 2, [TEST_PRODUCT_ID_BOOK_DIGITAL]);
    }

    protected function getKortingModelBookDigitalCommissions()
    {
        return new KortingModel('a', 9999, 1, 2, [TEST_PRODUCT_ID_BOOK_DIGITAL], [TEST_PRODUCT_ID_BOOK_DIGITAL => 123], [TEST_PRODUCT_ID_BOOK_DIGITAL => 321]);
    }

    protected function getKortingModelBookVariants()
    {
        return new KortingModel('a', 9999, 1, 2, [TEST_PRODUCT_ID_VARIANTS]);
    }

    public function testGetPrice1()
    {
        $this->requireSaleSetupPakkettenB();

        $kortingBookDigital = $this->getKortingModelBookDigital();
        $kortingVariants = $this->getKortingModelBookVariants();

        $this->kortingFactory->setModel(null);

        // no internal product
        $expected = 1000;
        $actual = $this->pricesHelper->getPrice1(TEST_NO_INTERNAL_PRODUCT_MODEL, null, true);
        $this->assertSame($expected, $actual);

        // no reduction model
        $expected = 10;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

        // no reductiom by different KortingModel
        $this->kortingFactory->setModel($kortingVariants);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

        // with reduction model
        $this->kortingFactory->setModel($kortingBookDigital);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $expected = 9;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);

        // variant, no reduction model
        $variant = 'a';
        $this->kortingFactory->setModel(null);
        $expected = 10;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        $variant = 'b';
        $expected = 11;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        // variant, with reduction model
        $this->kortingFactory->setModel($kortingVariants);
        $variant = 'a';
        $expected = 9;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $expected = 10;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        $variant = 'b';
        $expected = 10;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $expected = 11;
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        // minimal price
        $kortingModel = new KortingModel('a', 9999, 10**10, 10**10, [TEST_PRODUCT_ID_BOOK_DIGITAL]);
        $this->kortingFactory->setModel($kortingModel);
        $expected = $this->pricesHelper->getMinimalPrice1();
        $actual = $this->pricesHelper->getPrice1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
    }

    public function testGetCommission1()
    {
        $this->requireSaleSetupPakkettenB();

        // blacklist
        $this->pricesHelper->setSessionSaleProperty('is_blacklisted_affiliate', true, null, TEST_PRODUCT_ID_BOOK_DIGITAL);
        $expected = 0;
        $actual = $this->pricesHelper->getCommission1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);
        $this->pricesHelper->setSessionSaleProperty('is_blacklisted_affiliate', false, null, TEST_PRODUCT_ID_BOOK_DIGITAL);

        // no internal product model
        $expected = 50;
        $actual = $this->pricesHelper->getCommission1(TEST_NO_INTERNAL_PRODUCT_MODEL);
        $this->assertSame($expected, $actual);

        // based on commissions in kortingModel
        $this->kortingFactory->setModel($this->getKortingModelBookDigitalCommissions());
        $expected = 123;
        $actual = $this->pricesHelper->getCommission1(TEST_PRODUCT_ID_BOOK_DIGITAL);
        $this->assertSame($expected, $actual);

        // based on internalProductModel
        $this->kortingFactory->setModel($this->getKortingModelBookDigital());
        $expected = 20;
        $actual = $this->pricesHelper->getCommission1(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

    }

    public function testGetCurrentVariantPrice1()
    {
        $this->requireSaleSetupPakkettenB();

        $kortingBookDigital = $this->getKortingModelBookDigital();
        $kortingVariants = $this->getKortingModelBookVariants();

        // variant, with reduction model
        $this->kortingFactory->setModel($kortingVariants);
        $expected = 10;
        $actual = $this->pricesHelper->getCurrentVariantPrice1(TEST_PRODUCT_ID_VARIANTS, true);
        $this->assertSame($expected, $actual);
        $expected = 11;
        $actual = $this->pricesHelper->getCurrentVariantPrice1(TEST_PRODUCT_ID_VARIANTS, false);
        $this->assertSame($expected, $actual);

        // must not depend on sale_id (note that all sales should be stored in session)
        $sale_id = 'monkey';
        $expected = 10;
        $actual = $this->pricesHelper->getCurrentVariantPrice1(TEST_PRODUCT_ID_VARIANTS, true);
        $this->assertSame($expected, $actual);
    }

    public function testGetPriceN()
    {
        $this->requireSaleSetupPakkettenB();

        $kortingBookDigital = $this->getKortingModelBookDigital();
        $kortingVariants = $this->getKortingModelBookVariants();

        $this->kortingFactory->setModel(null);

        // no internal product
        $expected = 1997;
        $actual = $this->pricesHelper->getPriceN(TEST_NO_INTERNAL_PRODUCT_MODEL, null, true);
        $this->assertSame($expected, $actual);

        $kortingBookDigital = $this->getKortingModelBookDigital();
        $kortingVariants = $this->getKortingModelBookVariants();

        $this->kortingFactory->setModel(null);

        // no reduction model
        $expected = 110;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

        // no reductiom by different KortingModel
        $this->kortingFactory->setModel($kortingVariants);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

        // with reduction model
        $this->kortingFactory->setModel($kortingBookDigital);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $expected = 108;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);

        // variant, no reduction model
        $variant = 'a';
        $this->kortingFactory->setModel(null);
        $expected = 110;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        $variant = 'b';
        $expected = 111;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        // variant, with reduction model
        $this->kortingFactory->setModel($kortingVariants);
        $variant = 'a';
        $expected = 108;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $expected = 110;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        $variant = 'b';
        $expected = 109;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, true);
        $this->assertSame($expected, $actual);
        $expected = 111;
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_VARIANTS, $variant, false);
        $this->assertSame($expected, $actual);

        // minimal price
        $kortingModel = new KortingModel('a', 9999, 10**10, 10**10, [TEST_PRODUCT_ID_BOOK_DIGITAL]);
        $this->kortingFactory->setModel($kortingModel);
        $expected = $this->pricesHelper->getMinimalPriceN();
        $actual = $this->pricesHelper->getPriceN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, true);
        $this->assertSame($expected, $actual);
    }

    public function testGetCommissionN()
    {
        $this->requireSaleSetupPakkettenB();

        // blacklist
        $this->pricesHelper->setSessionSaleProperty('is_blacklisted_affiliate', true, null, TEST_PRODUCT_ID_BOOK_DIGITAL);
        $expected = 0;
        $actual = $this->pricesHelper->getCommissionN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);
        $this->pricesHelper->setSessionSaleProperty('is_blacklisted_affiliate', false, null, TEST_PRODUCT_ID_BOOK_DIGITAL);

        // no internal product model
        $expected = 51;
        $actual = $this->pricesHelper->getCommissionN(TEST_NO_INTERNAL_PRODUCT_MODEL);
        $this->assertSame($expected, $actual);

        // based on commissions in kortingModel
        $this->kortingFactory->setModel($this->getKortingModelBookDigitalCommissions());
        $expected = 321;
        $actual = $this->pricesHelper->getCommissionN(TEST_PRODUCT_ID_BOOK_DIGITAL);
        $this->assertSame($expected, $actual);

        // based on internalProductModel
        $this->kortingFactory->setModel($this->getKortingModelBookDigital());
        $expected = 120;
        $actual = $this->pricesHelper->getCommissionN(TEST_PRODUCT_ID_BOOK_DIGITAL, null, false);
        $this->assertSame($expected, $actual);

    }

    public function testGetCurrentVariantPriceN()
    {
        $this->requireSaleSetupPakkettenB();

        $kortingBookDigital = $this->getKortingModelBookDigital();
        $kortingVariants = $this->getKortingModelBookVariants();

        // variant, with reduction model
        $this->kortingFactory->setModel($kortingVariants);
        $expected = 109;
        $actual = $this->pricesHelper->getCurrentVariantPriceN(TEST_PRODUCT_ID_VARIANTS, true);
        $this->assertSame($expected, $actual);
        $expected = 111;
        $actual = $this->pricesHelper->getCurrentVariantPriceN(TEST_PRODUCT_ID_VARIANTS, false);
        $this->assertSame($expected, $actual);

        // must not depend on sale_id (note that all sales should be stored in session)
        $sale_id = 'monkey';
        $expected = 109;
        $actual = $this->pricesHelper->getCurrentVariantPriceN(TEST_PRODUCT_ID_VARIANTS, true);
        $this->assertSame($expected, $actual);
    }
    
    public function testSetSessionSaleProperty()
    {
        $this->requireSaleSetupPakkettenB();

        $this->assertNull($this->pricesHelper->getSessionSaleProperty('monkey', null, TEST_PRODUCT_ID_BOOK_DIGITAL));
        $this->pricesHelper->setSessionSaleProperty('monkey', 'banana', null, TEST_PRODUCT_ID_BOOK_DIGITAL);
        $this->assertSame('banana', $this->pricesHelper->getSessionSaleProperty('monkey', null, TEST_PRODUCT_ID_BOOK_DIGITAL));
    }

    protected function requireSaleSetupPakkettenB($saleId = 'variant')
    {
        global $sale_id;
        $sale_id = $saleId;
        if (!defined('$_COOKIE')) {
            define('$_COOKIE', []);
        }
        if (!defined('$_SESSION')) {
            define('$_SESSION', []);
        }

        $_COOKIE['pn'] = 'abc';
        $prevErrorReporting = error_reporting(0);
        require_once 'sale-setup-pakketten-variant-b.php';
        error_reporting($prevErrorReporting);
    }
}
