<?php

/**
 * Level Methods Hooks
 *
 * @package WishListMember
 */

namespace WishListMember;

/**
 * Level Methods Hooks trait
 */
trait Level_Methods_Hooks
{
    /**
     * Cron to trigger the expire/unexpire actions.
     *
     * @global \wpdb $wpdb
     */
    public function trigger_level_expiration_hooks()
    {
        global $wpdb;
        $wlmdb = &$wpdb;

        $levels = $this->get_option('wpm_levels');
        foreach ($levels as $level_id => $level) {
            $where_date = '';
            switch (wlm_arrval($level, 'expire_option')) {
                case '1': // fixed term.
                    /**
                     * Fixed term calendar period.
                     *
                     * @var string
                     */
                    $calendar = strtoupper(wlm_arrval($level, 'calendar'));
                    if (! in_array($calendar, ['DAYS', 'WEEKS', 'MONTHS', 'YEARS'], true)) {
                        $calendar = 'DAYS';
                    }

                    /**
                     * Date to be used in WHERE portion of SQL query for "fixed term" expiration
                     *
                     * @var string
                     */
                    $where_date = sprintf(
                        '"%s" >= date_add( rd.option_value, interval %d %s )',
                        wlm_date('Y-m-d H:i:s'),
                        esc_sql(wlm_arrval($level, 'expire')),
                        esc_sql(preg_replace('/[S]$/', '', $calendar))
                    );
                    // Continue to on date...
                case '2': // on date.
                    if (empty($where_date)) {
                        /**
                         * Date to be used in WHERE portion of SQL query for "on date" expiration
                         *
                         * @var string
                         */
                        $where_date = sprintf(
                            '"%s" >= "%s"', // always true.
                            wlm_date('Y-m-d H:i:s'),
                            wlm_date('Y-m-d H:i:s.u', strtotime(wlm_trim(wlm_arrval($level, 'expire_date')) . ' ' . wlm_timezone_string()))
                        );
                    }

                    /**
                     * Date to set our "expired" option to.
                     *
                     * This is generated by using the original $where_date and replacing
                     * everything from the start upto the ">=" with an empty string.
                     *
                     * @var string
                     */
                    $set_date = preg_replace('/^.+?>=/', '', $where_date);

                    /**
                     * INSERT portion of SQL query
                     *
                     * @var string
                     */
                    $query_insert = 'insert into %2$s ( userlevel_id, option_name, option_value ) select rd.userlevel_id, "expired", %4$s';

                    /**
                     * SELECT portion of SQL query
                     *
                     * @var string
                     */
                    $query_select = 'select ul.user_id';

                    /**
                     * FROM portion of SQL query
                     *
                     * @var string
                     */
                    $query_from = '`%1$s` as ul inner join `%2$s` as rd on ul.ID = rd.userlevel_id and ul.level_id = "%5$s" and rd.option_name = "registration_date" left join `%2$s` as exp on rd.userlevel_id = exp.userlevel_id and exp.option_name = "expired"';

                    /**
                     * WHERE portion of SQL query.
                     *
                     * @var string
                     */
                    $query_where = '%3$s and exp.ID is null';

                    /**
                     * SELECT query template
                     *
                     * @var string
                     */
                    $select_query_template = $query_select . ' from ' . $query_from . ' where ' . $query_where;

                    /**
                     * INSERT query template
                     *
                     * @var string
                     */
                    $insert_query_template = $query_insert . ' from ' . $query_from . ' where ' . $query_where;

                    /**
                     * Query template replacements.
                     *
                     * @var array
                     */
                    $replacements = [
                        $this->table_names->userlevels, // %1$s - wlm_userlevels table.
                        $this->table_names->userlevel_options, // %2$s - wlm_userlevel_options table.
                        $where_date, // %3$s - date for WHERE query.
                        $set_date, // %4$s - date to be used when inserting expired status.
                        esc_sql($level_id), // %5$s - membership level id.
                    ];

                    /**
                     * Full SELECT query to select memberships that are to be expired.
                     *
                     * @var string
                     */
                    $select_query = sprintf($select_query_template, ...array_values($replacements));

                    /**
                     * Full INSERT query to expire memberships.
                     *
                     * @var string
                     */
                    $insert_query = sprintf($insert_query_template, ...array_values($replacements));

                    // Get user ids to trigger for.
                    $user_ids = $wlmdb->get_col($select_query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                    if ($user_ids) {
                        // Mark levels as expired.
                        $wlmdb->query($insert_query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                        // Run expire action for users that are to be expired for the current level.
                        $this->do_expiration_action(true, $user_ids, $level_id);
                    }

                    /*
                     * Now we need to unexpire levels that are marked as expired but should not be.
                     * This happens when changed are made to the level expiration settings or a
                     * member's registration date.
                     */

                    /**
                     * Full SELECT query to select memberhips that are to be unexpired
                     *
                     * This is generated by using the original $select_query and replacing the following:
                     * - ">=" with "<"
                     * - "and exp.ID is null" with ""
                     * - "left join" with "inner join"
                     *
                     * @var string
                     */
                    $unexpire_select_query = str_replace(['>=', 'and exp.ID is null', 'left join'], ['<', '', 'inner join'], $select_query);

                    /**
                     * Full DELETE query to unexpire memberships
                     *
                     * This is generated by using the original $unexpire_select_query and replacing
                     * the value of $query_select with "delete exp"
                     *
                     * @var string
                     */
                    $unexpire_delete_query = str_replace($query_select, 'delete exp', $unexpire_select_query);

                    // Get users to trigger for.
                    $user_ids = $wlmdb->get_col($unexpire_select_query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                    if ($user_ids) {
                        // Mark levels as unexpired.
                        $wlmdb->query($unexpire_delete_query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                        // Run unexpire action for users that are to be unexpired for the current level.
                        $this->do_expiration_action(false, $user_ids, $level_id);
                    }

                    break;
                default: // no expiration. unexpire levels that are marked as expired for the current level.
                    $query_from = sprintf(
                        'from `%1$s` as ul join `%2$s` as ulo on ul.id=ulo.userlevel_id and ul.level_id="%3$s" and ulo.option_name="%4$s"',
                        $this->table_names->userlevels,
                        $this->table_names->userlevel_options,
                        esc_sql($level_id),
                        'expired'
                    );

                    // Get user ids to trigger for.
                    $user_ids = $wlmdb->get_col('select ul.user_id ' . $query_from); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                    if ($user_ids) {
                        // Mark levels as unexpired.
                        $wlmdb->query('delete ulo ' . $query_from); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared.

                        // Run unexpire action for users that are to be unexpired for the current level.
                        $this->do_expiration_action(false, $user_ids, $level_id);
                    }
            }
        }
    }

    /**
     * Called by wishlistmember_update_option_wpm_levels
     * Called by wishlistmember_add_option_wpm_levels
     * Called by wishlistmember_delete_option_wpm_levels
     *
     * @param mixed $new_value New levels.
     * @param mixed $old_value Old levels.
     */
    public function trigger_level_save_actions($new_value = [], $old_value = [])
    {
        // All levels were deleted.
        if ('wishlistmember_delete_option_wpm_levels' === current_action()) {
            /**
             * Fires when the wpm_levels option itself is deleted.
             *
             * @param string $level_id Always set to a value of "all".
             * @param array  $data     Empty array.
             */
            do_action('wishlistmember_level_deleted', 'all', []);
            return;
        }

        if (! is_array($new_value)) {
            $new_value = [];
        }
        if (! is_array($old_value)) {
            $old_value = [];
        }

        $old_value_keys = array_keys($old_value);
        $new_value_keys = array_keys($new_value);

        $deleted_levels = array_unique(array_diff($old_value_keys, $new_value_keys));
        $created_levels = array_unique(array_diff($new_value_keys, $old_value_keys));

        foreach ($deleted_levels as $level_id) {
            if (empty($level_id)) {
                continue;
            }
            /**
             * Fires when a level is deleted.
             *
             * @param string $level_id Level ID.
             * @param array  $data     Old Level Data.
             */
            do_action('wishlistmember_level_deleted', $level_id, $old_value[ $level_id ]);
        }

        foreach ($created_levels as $level_id) {
            if (empty($level_id)) {
                continue;
            }
            /**
             * Fires when a level is created.
             *
             * @param string $level_id Level ID.
             * @param array  $data     Level Data.
             */
            do_action('wishlistmember_level_created', $level_id, $new_value[ $level_id ]);
        }
    }

    /**
     * Trigger the expire/unexpire actions.
     *
     * @param boolean      $expired  True to expire, False to unexpire.
     * @param array        $user_ids Array of User IDs.
     * @param string|array $levels   A single Membership Level ID or an array of the same.
     */
    private function do_expiration_action($expired, $user_ids, $levels)
    {
        foreach ((array) $user_ids as $user_id) {
            /**
            * This action runs when a user's level expires/unexpires
            *
            * @param int    $user_id  User ID.
            * @param array  $level_id Level ID.
            */
            if ($expired) {
                do_action('wishlistmember_expire_user_levels', $user_id, (array) $levels);
            } else {
                do_action('wishlistmember_unexpire_user_levels', $user_id, (array) $levels);
            }
        }
    }
    /**
     * Cron that will complete the incomplete registrations which are stuck due to
     * their single cron event schedules being gone.
     *
     * @global \wpdb $wpdb
     */
    public function incomplete_registrations_checker()
    {
        global $wpdb;

        $incompletes = $this->get_incomplete_registrations();
        if (empty($incompletes)) {
            // No incomplete registrations.
            return;
        }
        // Initialize all level data.
        $levels = $this->get_option('wpm_levels');

        $inc_max = 0;
        foreach ($incompletes as $uid => $incomplete) {
            // Set maximum to 50 incomplete registrations.
            if ($inc_max >= 50) {
                break;
            }
            if (!isset($incomplete['wlm_incregnotification']['level']) || !isset($incomplete['email'])) {
                continue;
            }
            $inc_email  = $incomplete['email'];
            $inc_lvl_id = $incomplete['wlm_incregnotification']['level'];

            // If auto-create setting account setting together with the delay is enabled for the level the temp registration belongs.
            if (1 === (int) $levels[$inc_lvl_id]['autocreate_account_enable']  && ( 1 === (int) $levels[$inc_lvl_id]['autocreate_account_enable_delay'] )) {
                // Get incomplete registration's temp mail and use it to retrieve registration date.
                $inc_temp_mail = 'temp_' . md5($inc_email);
                $user          = get_user_by('login', $inc_temp_mail);

                // If registration date + delay is less than current date, run finish_incomplete_registration().
                $inc_delay_run_date = (int) gmdate(strtotime($user->user_registered)) + (int) $levels[$inc_lvl_id]['autocreate_account_delay'] * (int) $levels[$inc_lvl_id]['autocreate_account_delay_type'] * 60;
                $inc_current_day    = gmdate(time());

                if ($inc_delay_run_date < $inc_current_day) {
                    $inc_username_format = wlm_or(wlm_trim($levels[$inc_lvl_id]['autocreate_account_username']), $this->level_defaults['autocreate_account_username']);
                    $this->finish_incomplete_registration($inc_temp_mail, $inc_email, $inc_username_format);
                    // Only increase count if it gets here because there are levels which have incomplete registrations but not the auto-complete and delay settings enabled.
                    ++$inc_max;
                }
            }
        }
    }
}

// Register hooks.
add_action(
    'wishlistmember_register_hooks',
    function ($wlm) {
        add_action(
            'init',
            function () {
                // Schedule per-minute cron to process level expiration actions.
                if (! wp_next_scheduled('wishlistmember_run_level_expiration_actions')) {
                    wp_schedule_event(time(), 'wlm_minute', 'wishlistmember_run_level_expiration_actions');
                }

                // Schedule a cron every 15 minutes to process checking of incomplete registrations.
                if (! wp_next_scheduled('wishlistmember_incomplete_registrations_checker')) {
                    wp_schedule_event(time(), 'wlm_15minutes', 'wishlistmember_incomplete_registrations_checker');
                }
            }
        );
        add_action('wishlistmember_run_level_expiration_actions', [$wlm, 'trigger_level_expiration_hooks']);
        add_action('wishlistmember_incomplete_registrations_checker', [$wlm, 'incomplete_registrations_checker']);

        // Run actions for when a level is created or deleted.
        add_action('wishlistmember_update_option_wpm_levels', [$wlm, 'trigger_level_save_actions'], 10, 2);
        add_action('wishlistmember_add_option_wpm_levels', [$wlm, 'trigger_level_save_actions']);
        add_action('wishlistmember_delete_option_wpm_levels', [$wlm, 'trigger_level_save_actions']);
    }
);
