<?php

namespace WishListMember\Autoresponders;

if (! class_exists('\WLM_Sendlane')) {
    require_once wishlistmember_instance()->plugin_dir . '/extlib/wlm-sendlane.php';
}

class SendLane
{
    /**
     * Handle static method calls.
     *
     * This method is invoked when invoking inaccessible methods in a static context.
     * It checks if the API interface is available and calls the requested method
     * on the interface with the provided arguments.
     *
     * @param string $name The name of the method being called.
     * @param array  $args The arguments passed to the method.
     */
    public static function __callStatic($name, $args)
    {
        $interface = self::_interface();
        if ($interface->api()) {
            call_user_func_array([$interface, $name], $args);
        }
    }

    /**
     * Retrieves the singleton instance of the SendLaneInterface.
     *
     * This method ensures that only one instance of the SendLaneInterface
     * is created and reused throughout the class. It initializes the instance
     * on the first call and returns the same instance on subsequent calls.
     *
     * @return SendLaneInterface The singleton instance of the SendLaneInterface.
     */
    public static function _interface()
    {
        static $interface;
        if (! $interface) {
            $interface = new SendLaneInterface();
        }
        return $interface;
    }
}

class SendLaneInterface
{
    /**
     * Holds the settings for SendLane
     *
     * @var string
     */
    private $settings = '';

    /**
     * API key for authentication
     *
     * @var string|null
     */
    private $api_key = null;

    /**
     * API hash for authentication
     *
     * @var string|null
     */
    private $api_hash = null;

    /**
     * Subdomain for the SendLane API
     *
     * @var string|null
     */
    private $subdomain = null;

    /**
     * Constructor to initialize the SendLane API connection
     */
    public function __construct()
    {
        $this->sendlane_api = false;
        // Make sure that WLM active and SendLane connection is set.
        if (class_exists('\WLM_Sendlane')) {
            $this->settings = wlm_or(( new \WishListMember\Autoresponder('sendlane') )->settings, false);
            // Initilize sendlane api connection.
            if ($this->settings && ! empty($this->settings['api_key']) && ! empty($this->settings['api_hash']) && ! empty($this->settings['subdomain'])) {
                $this->sendlane_api = new \WLM_Sendlane($this->settings['api_key'], $this->settings['api_hash'], $this->settings['subdomain']);
                $lists_ret          = $this->sendlane_api->post('lists');
                if (isset($lists_ret['error'])) {
                    $this->sendlane_api = false;
                }
            }
        }
    }

    /**
     * Returns the SendLane API instance
     */
    public function api()
    {
        return $this->sendlane_api;
    }

    /**
     * Processes tags for the specified user levels and action (add, remove, etc.).
     *
     * @param  array|string $levels The levels to process tags for.
     * @param  string       $action The action to perform ('add', 'cancel', 'rereg', 'remove').
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function process_tags($levels, $action, $data)
    {
        if (! $this->sendlane_api) {
            return [
                'errstr' => 'Unable to process tags. No API Connection.',
                'errno'  => 1,
            ];
        }
        $levels = (array) $levels;
        if (count($levels) <= 0) {
            return [
                'errstr' => 'No Levels Found',
                'errno'  => 1,
            ];// No levels, no need to continue.
        }
        if (! isset($data['user_email']) || empty($data['user_email'])) {
            return [
                'errstr' => 'Email address not found',
                'errno'  => 1,
            ];
        }
        if (! in_array($action, ['add', 'cancel', 'rereg', 'remove'], true)) {
            return [
                'errstr' => 'Invalid action',
                'errno'  => 1,
            ];
        }

        $errors = [];
        // Add the tags for each level.
        foreach ((array) $levels as $level) {
            $error = [];

            if ('wishlistmember_payperpost_added' === current_action() || 'wishlistmember_payperpost_removed' === current_action()) {
                $level = 'payperpost-' . $level;
            }

            $apply_tags = isset($this->settings[ $level ][ $action ]['apply_tag']) ? $this->settings[ $level ][ $action ]['apply_tag'] : false;
            $apply_tags = ! empty($apply_tags) ? $apply_tags : false;
            $remove_tag = isset($this->settings[ $level ][ $action ]['remove_tag']) ? $this->settings[ $level ][ $action ]['remove_tag'] : false;
            $remove_tag = ! empty($remove_tag) ? $remove_tag : false;

            $list_add    = isset($this->settings[ $level ][ $action ]['list_add']) ? $this->settings[ $level ][ $action ]['list_add'] : false;
            $list_add    = ! empty($list_add) ? $list_add : false;
            $list_remove = isset($this->settings[ $level ][ $action ]['list_remove']) ? $this->settings[ $level ][ $action ]['list_remove'] : false;
            $list_remove = ! empty($list_remove) ? $list_remove : false;

            if ($list_add) {
                $param = [
                    'email'      => $data['user_email'],
                    'first_name' => $data['first_name'],
                    'last_name'  => $data['last_name'],
                    'list_id'    => $list_add,
                ];
                $ret   = $this->sendlane_api->post('list-subscriber-add', $param);
                if (isset($ret['error'])) {
                    $error['list_add'] = $ret['error']['messages'];
                }
            }

            if ($list_remove) {
                $param = [
                    'email'   => $data['user_email'],
                    'list_id' => $list_remove,
                ];
                $ret   = $this->sendlane_api->post('subscribers-delete', $param);
                // 406 means the email does not exists on the list.
                if (isset($ret['error']) && ! isset($ret['error'][406])) {
                    $error['list_remove'] = $ret['error']['messages'];
                }
            }

            if (! $apply_tags && ! $remove_tag) {
                continue; // Skip the rest of the loop.
            }

            // Now we can add or remove tags to record.
            if ($apply_tags) {
                $param = [
                    'email'   => $data['user_email'],
                    'tag_ids' => implode(',', $apply_tags),
                ];
                $ret   = $this->sendlane_api->post('tag-subscriber-add', $param);
                if (isset($ret['error'])) {
                    $error['apply_tag'] = $ret['error']['messages'];
                }
            }

            if ($remove_tag) {
                $param = [
                    'email'   => $data['user_email'],
                    'tag_ids' => implode(',', $remove_tag),
                ];
                $ret   = $this->sendlane_api->post('tag-subscriber-remove', $param);
                if (isset($ret['error'])) {
                    $error['remove_tag'] = $ret['error']['messages'];
                }
            }

            if (count($error)) {
                $errors[ $level ] = $error;
            }
        }
        return count($errors) ? wlm_maybe_serialize($errors) : true; // Success.
    }

    /**
     * Adds a data entry to the processing queue.
     *
     * @param array   $data    The data to be queued.
     * @param boolean $process Whether to process the queue immediately (default: true).
     */
    public function add_queue($data, $process = true)
    {
        $wishlist_api_queue_instance = new \WishListMember\API_Queue();
        $qname                       = 'sendlanear' . time();
        $data                        = wlm_maybe_serialize($data);
        $wishlist_api_queue_instance->add_queue($qname, $data, 'For Queueing');
        if ($process) {
            $this->process_queue();
        }
    }

    /**
     * Processes the queue of actions to be performed.
     *
     * @param integer $recnum The number of records to process (default: 10).
     * @param integer $tries  The number of retry attempts for each record (default: 3).
     */
    public function process_queue($recnum = 10, $tries = 3)
    {
        if (! $this->sendlane_api) {
            return;
        }
        $wishlist_api_queue_instance = new \WishListMember\API_Queue();
        $last_process                = get_option('WLM_AUTORESPONDER_SENDLANE_LastProcess');
        $current_time                = time();
        $tries                       = $tries > 1 ? (int) $tries : 3;
        $error                       = false;
        // Lets process every 10 seconds.
        if (! $last_process || ( $current_time - $last_process ) > 10) {
            $queues = $wishlist_api_queue_instance->get_queue('sendlanear', $recnum, $tries, 'tries,name');
            foreach ($queues as $queue) {
                $data = wlm_maybe_unserialize($queue->value);
                if ('new' === $data['action']) {
                    $res = $this->new_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('add' === $data['action']) {
                    $res = $this->add_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('remove' === $data['action']) {
                    $res = $this->remove_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('cancel' === $data['action']) {
                    $res = $this->cancel_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('rereg' === $data['action']) {
                    $res = $this->rereg_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('delete' === $data['action']) {
                    $res = $this->delete_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                }

                if (true !== $res) {
                    $d = [
                        'notes' => $res,
                        'tries' => $queue->tries + 1,
                    ];
                    $wishlist_api_queue_instance->update_queue($queue->ID, $d);
                    $error = true;
                } else {
                    $wishlist_api_queue_instance->delete_queue($queue->ID);
                    $error = false;
                }
            }
            // Save the last processing time.
            if ($error) {
                $current_time = time();
                if ($last_process) {
                    update_option('WLM_AUTORESPONDER_SENDLANE_LastProcess', $current_time);
                } else {
                    add_option('WLM_AUTORESPONDER_SENDLANE_LastProcess', $current_time);
                }
            }
        }
    }

    /**
     * Queues a new user for tag processing.
     *
     * @param integer|null $uid   The user ID.
     * @param array|null   $udata User data including email and names.
     */
    public function new_user_tags_hook_queue($uid = null, $udata = null)
    {
        // Part of the Fix for issue where Add To levels aren't being processed.
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        // Don't add the data into the queue if it's from a temp account.
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'new',
            'levels' => (array) $udata['wpm_id'],
            'data'   => $udata,
        ];
        $this->add_queue($data);
    }

    /**
     * Processes tags for a new user.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels to process tags for.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function new_user_tags_hook($uid, $levels, $data)
    {
        $tempacct = 'temp_' . md5($data['orig_email']) === $data['email'];
        if ($tempacct) {
            return; // If temp account used by sc, do not process.
        }
        return $this->process_tags($levels, 'add', $data);
    }

    /**
     * Queues a user for tag addition when added to levels.
     *
     * @param integer $uid       The user ID.
     * @param string  $addlevels The levels to which the user is being added.
     */
    public function add_user_tags_hook_queue($uid, $addlevels = '')
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }

        $udata               = [];
        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'add',
            'levels' => $addlevels,
            'data'   => $udata,
        ];
        // Fix for issue where Add To levels aren't being processed.
        // If the data is from a temp account then add it to the queue API and don't process it for now.
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            $this->add_queue($data, 0);
        } elseif (isset(wlm_post_data()['SendMail'])) {
            // This elseif condition fixes the issue where members who are added via.
            // WLM API aren't being processed by the Integration.
            $this->add_queue($data, 0);
        } else {
            $this->add_queue($data);
        }
    }

    /**
     * Processes tags for a user when added to levels.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels to which the user is being added.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function add_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        // Make sure that info are updated.
        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'add', $data);
    }

    /**
     * Queues a user for tag removal when removed from levels.
     *
     * @param integer $uid           The user ID.
     * @param string  $removedlevels The levels from which the user is being removed.
     */
    public function remove_user_tags_hook_queue($uid, $removedlevels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $removedlevels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'remove',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes tags for a user when removed from levels.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels from which the user is being removed.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function remove_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'remove', $data);
    }

    /**
     * Queues a user for tag cancellation when cancelled from levels.
     *
     * @param integer $uid          The user ID.
     * @param string  $cancellevels The levels from which the user is being cancelled.
     */
    public function cancel_user_tags_hook_queue($uid, $cancellevels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $cancellevels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'cancel',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes tags for a user when cancelled from levels.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels from which the user is being cancelled.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function cancel_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'cancel', $data);
    }

    /**
     * Queues a user for re-registration when re-registered from levels.
     *
     * @param integer $uid    The user ID.
     * @param string  $levels The levels from which the user is being re-registered.
     */
    public function rereg_user_tags_hook_queue($uid, $levels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $levels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'rereg',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes tags for a user when re-registered from levels.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels from which the user is being re-registered.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function rereg_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'rereg', $data);
    }

    /**
     * Queues a user for deletion when deleted from levels.
     *
     * @param integer $uid The user ID.
     */
    public function delete_user_hook_queue($uid)
    {
        if (! $this->sendlane_api) {
            return;
        }

        $levels = wishlistmember_instance()->get_membership_levels($uid);
        foreach ($levels as $key => $lvl) {
            if (false !== strpos($lvl, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (! is_array($levels) || count($levels) <= 0) {
            return; // Lets return if no level was found.
        }

        $user = get_userdata($uid);
        if (! $user) {
            return;
        }

        $udata               = [];
        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'delete',
            'levels' => $levels,
            'data'   => $udata,
        ];
        $this->add_queue($data);
        return;
    }

    /**
     * Processes tags for a user when deleted from levels.
     *
     * @param  integer      $uid    The user ID.
     * @param  array|string $levels The levels from which the user is being deleted.
     * @param  array        $data   User data including email and names.
     * @return array|boolean An array with error details or true on success.
     */
    public function delete_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $levels = (array) $levels;
        return $this->process_tags($levels, 'remove', $data);
    }
}
