<?php

namespace WishListMember\Infusionsoft;

class WLM_Infusionsoft
{
    /**
     * WLM Infusionsoft Connection Object
     *
     * @var \WishListMember\Infusionsoft\WLM_Infusionsoft_Connection
     */
    private $ifsdk_con = null;

    /**
     * WLM Infusionsoft Object
     *
     * @var \WishListMember\Infusionsoft\Infusionsoft
     */
    private $ifservices = null;

    /**
     * Infusionsoft Integration Type
     *
     * Can be payment or autoresponder
     *
     * @var string
     */
    private $integration_type = 'payment';

    /**
     * Infusionsoft Token Suffix used for saving token information
     *
     * @var string
     */
    private $token_suffix = '';

    /**
     * Autoresponder Integration client ID
     *
     * @var string
     */
    private static $client_id_autoresponder = 'R8uVj4q3fmfADA9GxLFZ9TtpTG3F7XYsFgdxQ6KsjRVn0ohV';

    /**
     * Payment Integration client ID
     *
     * @var string
     */
    private static $client_id_payment = 'xGYxxcJQObf4RFmFinQ5Cf8sitMuN9eSkzQg4jFFNJMKydoY';

    public function __construct($type = 'payment', $api_key = '', $debug = false)
    {
        $this->integration_type = $type;
        $this->token_suffix = '_' . $this->integration_type;
        switch ($type) {
            case 'payment':
                $client_id = self::$client_id_payment;
                break;
            case 'autoresponder':
                $client_id = self::$client_id_autoresponder;
                break;
            default:
                throw new \Exception('Invalid Integration type');
                break;
        }

        $this->ifsdk_con = new WLM_Infusionsoft_Connection($type, $client_id);
        if ($debug) {
            $this->ifsdk_con->enable_logging($debug);
        }
        // If no API Key is set, we are using OAuth.
        if ($api_key) {
            $this->ifsdk_con->set_connection($api_key);
        }
    }

    public function process_oauth_authorization($code)
    {
        $token = $this->ifsdk_con->get_token_from_code($code);

        if (!$token) {
            return false;
        }

        // If new token was created, were successfully authenticated.
        $code = [
            'code' => $code,
            'time' => time(),
            'requestdate' => current_datetime()->format('Y-m-d H:i:s'),
        ];

        $scope = $token->extraInfo['scope'] ?? '';
        $scope = explode('|', $scope);
        $app = $scope[1] ?? $token->scope;
        wishlistmember_instance()->save_option('keap_oauth_app' . $this->token_suffix, $app);

        wishlistmember_instance()->save_option('keap_oauth_code' . $this->token_suffix, $code);
        wishlistmember_instance()->save_option('keap_oauth_token' . $this->token_suffix, $token);
        /**
         * We'll create a transient for 23 hours.
         * This acts as marker for us when the token needs to be refreshed.
         * This is to prevent the token from being refreshed too often.
         */
        set_transient('keap_token_refresh' . $this->token_suffix, $token, 60 * 60 * 23);

        // Lets remove the old api key and do a backup.
        $wlm_apikey_option = 'payment' === $this->integration_type ? 'isapikey' : 'auto_isapikey';
        $api_key = wishlistmember_instance()->get_option($wlm_apikey_option);
        if ($api_key) {
            wishlistmember_instance()->save_option($wlm_apikey_option, false);
            wishlistmember_instance()->save_option("_{$wlm_apikey_option}_bak", $api_key);
        }
        if ('autoresponder' === $this->integration_type) {
                // For autoresponder, apikey is also saved in autoresponder settings.
            $ar = wishlistmember_instance()->get_option('Autoresponders');
            if (isset($ar['infusionsoft']['iskey'])) {
                wishlistmember_instance()->save_option("_{$wlm_apikey_option}_bak", $ar['infusionsoft']['iskey']);
            }
            // Lets  remove api key from autoresponder settings.
            $ar['infusionsoft']['iskey'] = '';
            wishlistmember_instance()->save_option('Autoresponders', $ar);
        }
        return true;
    }

    public function get_oauth_authorize_url()
    {
        return $this->ifsdk_con->get_oauth_authorize_url();
    }

    public function is_api_connected()
    {
        return $this->ifsdk_con->get_connection() ? true : false;
    }

    public function create_contact($cdetails, $optin_reason = 'Contact Opted In through the WLM API')
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        $contact_id = $this->ifservices->contacts('xml')->add($cdetails);
        if (! empty($cdetails['Email'])) {
            $this->ifservices->emails('xml')->optIn(
                $cdetails['Email'],
                $optin_reason
            );
        }
        return $contact_id;
    }

    public function get_contact_details($contact_id)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }

        $fields = ['Id', 'FirstName', 'LastName', 'Email', 'Company', 'StreetAddress1', 'StreetAddress2', 'City', 'State', 'PostalCode', 'Country'];
        return $this->ifservices->contacts('xml')->load($contact_id, $fields);
    }

    public function get_contactid_by_email($email)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        $c = $this->ifservices->contacts('xml')->findByEmail($email, ['Id']);
        return $c ? $c[0]['Id'] : false;
    }

    public function assign_followup_sequence($cid, $campid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        return $this->ifservices->contacts('xml')->addToCampaign($cid, $campid);
    }

    public function remove_followup_sequence($cid, $campid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        return $this->ifservices->contacts('xml')->removeFromCampaign($cid, $campid);
    }

    public function get_contactid_by_invoice($invid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        $invoice = $this->ifservices->data('xml')->findByField('Invoice', 1, 0, 'Id', $invid, ['ContactId']);

        if ($invoice) {
            return isset($invoice[0]['ContactId']) ? $invoice[0]['ContactId'] : false;
        } else {
            return false;
        }
    }

    public function optin_contact_email($email, $reason = 'Contact Opted In through the WLM API')
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        return $this->ifservices->emails('xml')->optin(
            [
                'email' => $email,
                'optinReason' => $reason,
            ]
        );
    }

    public function get_tag_categories()
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        $page = 0;
        $tags_category = [];
        while ($page >= 0) { // Lets fetch records by 1000 per call/page.
            $tcategory = $this->ifservices->data('xml')->query('ContactGroupCategory', 1000, $page, ['Id' => '%'], ['Id', 'CategoryName', 'CategoryDescription'], '', false);
            if ($tcategory) {
                foreach ($tcategory as $id => $data) {
                    $tags_category[ $data['Id'] ] = $data['CategoryName'];
                }
                ++$page;
            } else {
                $page = -1;
            }
        }
        return $tags_category;
    }

    public function get_tags()
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        $page = 0;
        $tags = [];
        while ($page >= 0) { // Lets fetch records by 1000 per call/page.
            $t = $this->ifservices->data('xml')->query('ContactGroup', 1000, $page, ['Id' => '%'], ['Id', 'GroupName', 'GroupCategoryId'], '', false);
            if ($t) {
                foreach ($t as $id => $data) {
                    $tags[ $data['GroupCategoryId'] ][] = [
                        'Id' => $data['Id'],
                        'Name' => $data['GroupName'],
                    ];
                }
                ++$page;
            } else {
                $page = -1;
            }
        }
        asort($tags);
        return $tags;
    }

    public function tag_contact($contactid, $tagid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        try {
            $ret = $this->ifservices->contacts('xml')->addToGroup($contactid, $tagid);
        } catch (\Exception $e) {
            return null;
        }
        if ($ret) {
            return $ret;
        } else {
            $t = [
                'errno' => 1,
                'errstr' => 'Apply Tag Error',
            ];
        }
    }

    public function untag_contact($contactid, $tagid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return null;
        }
        try {
            $ret = $this->ifservices->contacts('xml')->removeFromGroup($contactid, $tagid);
        } catch (\Exception $e) {
            return null;
        }
        if ($ret) {
            return $ret;
        } else {
            $t = [
                'errno' => 1,
                'errstr' => 'Remove Tag Error',
            ];
        }
    }

    public function get_orderid_job($orderid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'JobTitle', 'ContactId', 'StartDate', 'DueDate', 'JobNotes', 'ProductId', 'JobStatus', 'DateCreated', 'JobRecurringId', 'OrderType', 'OrderStatus'];
        $res = $this->ifservices->data('xml')->findByField('Job', 1, 0, 'Id', $orderid, $fields);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_invoice_details($invoiceid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'ContactId', 'JobId', 'DateCreated', 'TotalDue', 'PayStatus', 'CreditStatus', 'RefundStatus', 'PayPlanStatus', 'InvoiceType', 'ProductSold'];
        $res = $this->ifservices->data('xml')->findByField('Invoice', 1, 0, 'Id', $invoiceid, $fields);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_jobid_invoice($jobid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'ContactId', 'DateCreated', 'TotalDue', 'JobId', 'PayStatus', 'CreditStatus', 'RefundStatus', 'PayPlanStatus', 'InvoiceType', 'ProductSold'];
        $res = $this->ifservices->data('xml')->findByField('Invoice', 1, 0, 'JobId', $jobid, $fields);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_cidpid_recurringorder($contactid, $pid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'ContactId', 'OriginatingOrderId', 'SubscriptionPlanId', 'ProductId', 'StartDate', 'EndDate', 'LastBillDate', 'NextBillDate', 'ReasonStopped', 'Status'];
        $query = [
            'ContactId' => $contactid,
            'ProductId' => $pid,
        ];
        $res = $this->ifservices->data('xml')->query('RecurringOrder', 1, 0, $query, $fields, '', false);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_subscriptionid_recurringorder($subscriptionid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'ContactId', 'OriginatingOrderId', 'SubscriptionPlanId', 'ProductId', 'StartDate', 'EndDate', 'LastBillDate', 'NextBillDate', 'ReasonStopped', 'Status'];
        $query = ['Id' => $subscriptionid];
        $res = $this->ifservices->data('xml')->query('RecurringOrder', 1, 0, $query, $fields, '', false);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_cidjobid_recurringorder($contactid, $jobid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'ContactId', 'OriginatingOrderId', 'SubscriptionPlanId', 'ProductId', 'StartDate', 'EndDate', 'LastBillDate', 'NextBillDate', 'ReasonStopped', 'Status'];
        $query = [
            'ContactId' => $contactid,
            'OriginatingOrderId' => $jobid,
        ];
        $res = $this->ifservices->data('xml')->query('RecurringOrder', 1, 0, $query, $fields, '', false);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_invoice_payments($invoiceid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'InvoiceId', 'Amt', 'PayStatus', 'PaymentId', 'SkipCommission'];
        $query = ['InvoiceId' => $invoiceid];
        $page = 0;
        $payments = [];
        while ($page >= 0) { // Lets fetch records by 1000 per call/page.
            $res = $this->ifservices->data('xml')->query('InvoicePayment', 1000, $page, $query, $fields, '', false);
            if ($res) {
                foreach ($res as $id => $data) {
                    $payments[ $id ] = $data;
                }
                ++$page;
            } else {
                $page = -1;
            }
        }

        if ($payments) {
            return $payments;
        } else {
            return false;
        }
    }

    public function get_subscriptionid_jobs($subscriptionid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'JobTitle', 'ContactId', 'JobRecurringId', 'ProductId'];
        $query = ['JobRecurringId' => $subscriptionid];
        $page = 0;
        $jobs = [];
        while ($page >= 0) { // Lets fetch records by 1000 per call/page.
            $res = $this->ifservices->data('xml')->query('Job', 1000, $page, $query, $fields, '', false);
            if ($res) {
                foreach ($res as $id => $data) {
                    $jobs[ $id ] = $data;
                }
                ++$page;
            } else {
                $page = -1;
            }
        }

        if ($jobs) {
            return $jobs;
        } else {
            return false;
        }
    }

    public function get_invoice_payplan($invoiceid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'InvoiceId', 'AmtDue', 'DateDue', 'InitDate', 'StartDate'];
        $res = $this->ifservices->data('xml')->findByField('PayPlan', 1, 0, 'InvoiceId', $invoiceid, $fields);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }

    public function get_payplan_items($payplanid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'PayPlanId', 'DateDue', 'AmtDue', 'Status'];
        $query = ['PayPlanId' => $payplanid];
        $page = 0;
        $ppi = [];
        while ($page >= 0) { // Lets fetch records by 1000 per call/page.
            $res = $this->ifservices->data('xml')->query('PayPlanItem', 1000, $page, $query, $fields, '', false);
            if ($res) {
                foreach ($res as $id => $data) {
                    $ppi[ $id ] = $data;
                }
                ++$page;
            } else {
                $page = -1;
            }
        }

        if ($ppi) {
            return $ppi;
        } else {
            return false;
        }
    }

    public function get_product_sku($pid)
    {
        $this->ifservices = $this->ifsdk_con->get_connection();
        if (! $this->ifservices) {
            return false;
        }
        $fields = ['Id', 'Sku', 'ProductName'];
        $res = $this->ifservices->data('xml')->findByField('Product', 1, 0, 'Id', $pid, $fields);
        if ($res) {
            return $res[0];
        } else {
            return false;
        }
    }
}
