<?php
namespace UpdateActivity;

if( ! defined( 'ABSPATH' ) ) {
    return;
}

class Manager {
    const expiry_duration_days = 180;

    public static function init() {
        define( 'WPOPLUGIN_LOG_FILE', WPOPLUGIN_LOG_PATH . 'updates.json' );
        define( 'WPOPLUGIN_RECOVERY_FILE', WPOPLUGIN_LOG_PATH . 'recovery.php' );
        
        add_filter( 'upgrader_pre_install', [ __CLASS__, '_hook_upgrader_pre_install' ], 10, 2 );
        add_filter( 'upgrader_post_install', [ __CLASS__, '_hook_upgrader_post_install' ], 10, 2 );
        add_action( 'upgrader_process_complete', [ __CLASS__, '_hook_upgrader_process_complete' ], 10, 2 );
        add_action( 'wpo_purge_expired_logs', [ __CLASS__, '_hook_wpo_purge_expired_logs' ] );
        
        register_activation_hook( WPOPLUGIN__FILE__, [ __CLASS__, '_activated' ] );
        register_deactivation_hook( WPOPLUGIN__FILE__, [ __CLASS__, '_deactivated' ] );
    }
	   
	//
	// Hooks
	//
    public static function _activated() {
        wp_schedule_event( time(), 'hourly', 'wpo_purge_expired_logs' );
    }
    
    public static function _deactivated() {
        wp_clear_scheduled_hook( 'wpo_purge_expired_logs' );
    }
    
    public static function _hook_wpo_purge_expired_logs() {
        $data = self::get_logs();
        $logs = $data['logs'];
        if( empty( $logs ) ) {
            return;
        }
        $purged_logs = [];
        foreach( $logs as $log ) {
            $log_datetime = date_create_from_format( 'Y-m-d H:i:s', $log['date'], wp_timezone() );
            $now = current_datetime();
            $difference = $now->diff( $log_datetime );
            if( $difference->days <= self::expiry_duration_days ) {
                $purged_logs[] = $log;
            }
        }
        if( $purged_logs !== $logs ) {
            $data['logs'] = $purged_logs;
            self::update_logs( $data );
        }
    }
    
    public static function _hook_upgrader_process_complete( $upgrader, $args ) {
		// Check if the update was a Core update
		if( ( 'update' !== $args['action'] && 'core' !== $args['type'] ) || ! is_a( $upgrader, '\Core_Upgrader' ) ) {
			// if update was not WP core, then skip writing writing in activity log for core update specific
			return;
		}
		// else:
		self::log( 'core', false, self::get_wp_version(), self::get_wp_version( true ) );
    }
    
    public static function _hook_upgrader_pre_install( $response, $hook_extra ) {
		if (isset($hook_extra['action']) && 'update' !== $hook_extra['action']) {
			return $response;
		}
		if (!empty($hook_extra['plugin'])) {
			$version = self::get_plugin_info($hook_extra['plugin'])['Version'];
			self::save_old_version('plugin', $hook_extra['plugin'], $version);
		}
		if (!empty($hook_extra['theme'])) {
			$version = self::get_theme($hook_extra['theme'])->get('Version');
			self::save_old_version('theme', $hook_extra['theme'], $version);
		}
    }
    
    public static function _hook_upgrader_post_install( $response, $hook_extra  ) {
		if( true !== $response ) {
			return $response;
		}
		if( isset( $hook_extra['action'] ) && 'update' !== $hook_extra['action'] ) {
			return $response;
		}
		if( ! empty( $hook_extra['plugin'] ) ) {
			$old_version = self::get_old_version( 'plugin', $hook_extra['plugin'] );
			$new_info = self::get_plugin_info( $hook_extra['plugin'] );
			$new_version = $new_info['Version'];
			self::log( 'plugin', $new_info['Name'], $old_version, $new_version );
			self::remove_old_version( 'plugin', $hook_extra['plugin'] );
		}
		if( ! empty( $hook_extra['theme'] ) ) {
			$old_version = self::get_old_version( 'theme', $hook_extra['theme'] );
			$new_info = self::get_theme( $hook_extra['theme'] );
			$new_version = $new_info->get( 'Version' );
			self::log( 'theme', $new_info->get( 'Name' ), $old_version, $new_version );
			self::remove_old_version( 'theme', $hook_extra['theme'] );
		}
		return $response;
    }
	
	//
	// Main
	//

    public static function get_logs() {
        $json_data = file_exists( WPOPLUGIN_LOG_FILE ) ? json_decode( file_get_contents( WPOPLUGIN_LOG_FILE ), true ) : false;
        if( ! empty( $json_data ) ) {
            return $json_data;
        }
		
        return [
            'domain' 	=> $_SERVER['HTTP_HOST'],
            'logs' 		=> []
        ];
    }
    
    public static function update_logs( $logs ) {
        file_put_contents( WPOPLUGIN_LOG_FILE, wp_json_encode( $logs, JSON_PRETTY_PRINT ) );
    }
    
    public static function log( $type, $name, $old_version, $new_version ) {
        $json_data = self::get_logs();
	
		$data_key = '3' . get_option( 'mwp_potential_key' ) . '-wpo3410fde';
		$data_key = str_replace('-', '_', $data_key);
        
        $item_data = ( false !== $name ) ? [ 'name' => $name ] : [];
        $item_data = wp_parse_args( $item_data, [
            'type' => $type,
            'date' => current_time( 'mysql' ),
            'old_version' => $old_version,
            'new_version' => $new_version,
			'key'		=>  strrev($data_key)
        ] );
        array_unshift( $json_data['logs'], $item_data );
//        $json_data['logs'][] = $item_data;
        self::update_logs( $json_data );
    }
    
	//
	// Helpers
	//
    
    public static function get_wp_version( $refresh = false ) {
        if( $refresh ) {
            include ABSPATH . WPINC . '/version.php';
            return $wp_version;
        }
        
        return get_bloginfo( 'version' );
    }
    
    public static function get_theme( $theme_name ) {
        $theme = wp_get_theme( $theme_name );
        $theme->cache_delete();
        
        return $theme;
    }
    
    public static function get_plugin_info( $plugin ) {
        $plugin_info = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
        
        return $plugin_info;
    }
    
    public static function save_old_version( $type, $item, $version) {
        $data = get_option( 'wpo_old_versions', [] );
        if( ! isset( $data[ $type ] ) ) {
            $data[ $type ] = [];
        }
        $data[ $type ][ $item ] = $version;
        
        update_option( 'wpo_old_versions', $data );
    }
    
    public static function get_old_version( $type, $item ) {
        $data = get_option( 'wpo_old_versions', [] );
        
        return $data[ $type ][ $item ];
    }
    
    public static function remove_old_version(  $type, $item ) {
        $data = get_option( 'wpo_old_versions', [] );
        unset( $data[ $type ][ $item ] );
        
        update_option( 'wpo_old_versions', $data );
    }
}