<?php
/**
 * @package		OpenCart
 * @author		Daniel Kerr
 * @copyright	Copyright (c) 2005 - 2017, OpenCart, Ltd. (https://www.opencart.com/)
 * @license		https://opensource.org/licenses/GPL-3.0
 * @link		https://www.opencart.com
*/

/**
* Loader class
*/
final class Loader {
	protected $registry;

	/* NeoSeo Firestore BEGIN */
	protected $theme_image_sizes;
	/* NeoSeo Firestore END */

	/**
	 * Constructor
	 *
	 * @param	object	$registry
 	*/
	public function __construct($registry) {
		$this->registry = $registry;
	}

	/**
	 * 
	 *
	 * @param	string	$route
	 * @param	array	$data
	 *
	 * @return	mixed
 	*/	
	public function controller($route, $data = array()) {
		// Sanitize the call
		$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);
		
		// Keep the original trigger
		$trigger = $route;
		
		// Trigger the pre events
		$result = $this->registry->get('event')->trigger('controller/' . $trigger . '/before', array(&$route, &$data));
		
		// Make sure its only the last event that returns an output if required.
		if ($result != null && !$result instanceof Exception) {
			$output = $result;
		} else {
			$action = new Action($route);
			$output = $action->execute($this->registry, array(&$data));
		}
		
		// Trigger the post events
		$result = $this->registry->get('event')->trigger('controller/' . $trigger . '/after', array(&$route, &$data, &$output));
		
		if ($result && !$result instanceof Exception) {
			$output = $result;
		}

		if (!$output instanceof Exception) {
			return $output;
		}
	}

	/**
	 * 
	 *
	 * @param	string	$route
 	*/	
	public function model($route) {
		// Sanitize the call
		$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);
		
		if (!$this->registry->has('model_' . str_replace('/', '_', $route))) {
			$file  = DIR_APPLICATION . 'model/' . $route . '.php';
			$class = 'Model' . preg_replace('/[^a-zA-Z0-9]/', '', $route);
			
			if (is_file($file)) {
				include_once(modification($file));
	
				$proxy = new Proxy();
				
				// Overriding models is a little harder so we have to use PHP's magic methods
				// In future version we can use runkit
				foreach (get_class_methods($class) as $method) {
					$proxy->{$method} = $this->callback($this->registry, $route . '/' . $method);
				}
				
				$this->registry->set('model_' . str_replace('/', '_', (string)$route), $proxy);
			} else {
				throw new \Exception('Error: Could not load model ' . $route . '!');
			}
		}
	}

	/**
	 * 
	 *
	 * @param	string	$route
	 * @param	array	$data
	 *
	 * @return	string
 	*/
	public function view($route, $data = array()) {
		// Sanitize the call
		$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);
		
		// Keep the original trigger
		$trigger = $route;
		
		// Template contents. Not the output!
		$code = '';
		
		// Trigger the pre events
		$result = $this->registry->get('event')->trigger('view/' . $trigger . '/before', array(&$route, &$data, &$code));
		
		// Make sure its only the last event that returns an output if required.
		if ($result && !$result instanceof Exception) {
			$output = $result;
		} else {
			$template = new Template($this->registry->get('config')->get('template_engine'));
				
			foreach ($data as $key => $value) {
				$template->set($key, $value);
			}


			/* NeoSeo Firestore BEGIN */
			$current_theme = trim($this->registry->get('config')->get('config_theme'));
			if(!defined('HTTPS_CATALOG') && $this->registry->get('config')->get('theme_' . $current_theme . '_status') && strpos($this->registry->get('config')->get('config_theme'),'neoseo_firestore') === 0 ){
				$theme = $this->registry->get('config')->get('template_directory') ;
				$explode = explode('/',$theme);
				if ($this->registry->get('config')->is_mobile() && $explode[0] != 'default') {
					$in_row_products_count = $this->registry->get('config')->get('theme_' . $current_theme . '_products_in_row_mobile');
				} else {
					$in_row_products_count = $this->registry->get('config')->get('theme_' . $current_theme . '_products_in_row_desktop');
				}
				if(!($in_row_products_count > 0)) $in_row_products_count = 1;
				$template->set('in_row_products_count',$in_row_products_count);
				if(!is_array($this->theme_image_sizes)){
				$template_name = $this->registry->get('config')->get('config_theme');
					$this->theme_image_sizes = array(
						'image_category_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_category_width'),
						'image_category_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_category_height'),
						'image_manufacturer_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_manufacturer_width'),
						'image_manufacturer_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_manufacturer_height'),
						'image_thumb_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_thumb_width'),
						'image_thumb_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_thumb_height'),
						'image_popup_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_popup_width'),
						'image_popup_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_popup_height'),
						'image_zoom_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_zoom_width'),
						'image_zoom_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_zoom_height'),
						'image_product_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_product_width'),
						'image_product_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_product_height'),
						'image_additional_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_additional_width'),
						'image_additional_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_additional_height'),
						'image_related_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_related_width'),
						'image_related_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_related_height'),
						'image_compare_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_compare_width'),
						'image_compare_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_compare_height'),
						'image_wishlist_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_wishlist_width'),
						'image_wishlist_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_wishlist_height'),
						'image_cart_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_cart_width'),
						'image_cart_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_cart_height'),
						'image_location_width' => $this->registry->get('config')->get('theme_'.$template_name.'_image_location_width'),
						'image_location_height' => $this->registry->get('config')->get('theme_'.$template_name.'_image_location_height'),
						'configblog_image_related_width' => $this->registry->get('config')->get('configblog_image_related_width'),
						'configblog_image_related_height' => $this->registry->get('config')->get('configblog_image_related_height'),
						'configblog_image_category_width' => $this->registry->get('config')->get('configblog_image_category_width'),
						'configblog_image_category_height' => $this->registry->get('config')->get('configblog_image_category_height'),
						'configblog_image_article_width' => $this->registry->get('config')->get('configblog_image_article_width'),
						'configblog_image_article_height' => $this->registry->get('config')->get('configblog_image_article_height'),
					);
				}
				foreach ($this->theme_image_sizes as $key => $value) {
					$template->set($key, $value);
				}
			}
			/* NeoSeo Firestore END */
			$output = $template->render($this->registry->get('config')->get('template_directory') . $route, $code);
		}
		
		// Trigger the post events
		$result = $this->registry->get('event')->trigger('view/' . $trigger . '/after', array(&$route, &$data, &$output));
		
		if ($result && !$result instanceof Exception) {
			$output = $result;
		}
		
		return $output;
	}

	/**
	 * 
	 *
	 * @param	string	$route
 	*/
	public function library($route) {
		// Sanitize the call
		$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);
			
		$file = DIR_SYSTEM . 'library/' . $route . '.php';
		$class = str_replace('/', '\\', $route);

		if (is_file($file)) {
			include_once(modification($file));

			$this->registry->set(basename($route), new $class($this->registry));
		} else {
			throw new \Exception('Error: Could not load library ' . $route . '!');
		}
	}

	/**
	 * 
	 *
	 * @param	string	$route
 	*/	
	public function helper($route) {
		$file = DIR_SYSTEM . 'helper/' . preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route) . '.php';

		if (is_file($file)) {
			include_once(modification($file));
		} else {
			throw new \Exception('Error: Could not load helper ' . $route . '!');
		}
	}

	/**
	 * 
	 *
	 * @param	string	$route
 	*/	
	public function config($route) {
		$this->registry->get('event')->trigger('config/' . $route . '/before', array(&$route));
		
		$this->registry->get('config')->load($route);
		
		$this->registry->get('event')->trigger('config/' . $route . '/after', array(&$route));
	}

	/**
	 * 
	 *
	 * @param	string	$route
	 * @param	string	$key
	 *
	 * @return	array
 	*/
	public function language($route, $key = '') {
		// Sanitize the call
		$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);
		
		// Keep the original trigger
		$trigger = $route;
				
		$result = $this->registry->get('event')->trigger('language/' . $trigger . '/before', array(&$route, &$key));
		
		if ($result && !$result instanceof Exception) {
			$output = $result;
		} else {
			$output = $this->registry->get('language')->load($route, $key);
		}
		
		$result = $this->registry->get('event')->trigger('language/' . $trigger . '/after', array(&$route, &$key, &$output));
		
		if ($result && !$result instanceof Exception) {
			$output = $result;
		}
				
		return $output;
	}
	
	protected function callback($registry, $route) {
		return function($args) use($registry, $route) {
			static $model;
			
			$route = preg_replace('/[^a-zA-Z0-9_\/]/', '', (string)$route);

			// Keep the original trigger
			$trigger = $route;
					
			// Trigger the pre events
			$result = $registry->get('event')->trigger('model/' . $trigger . '/before', array(&$route, &$args));
			
			if ($result && !$result instanceof Exception) {
				$output = $result;
			} else {
				$class = 'Model' . preg_replace('/[^a-zA-Z0-9]/', '', substr($route, 0, strrpos($route, '/')));
				
				// Store the model object
				$key = substr($route, 0, strrpos($route, '/'));
				
				if (!isset($model[$key])) {
					$model[$key] = new $class($registry);
				}
				
				$method = substr($route, strrpos($route, '/') + 1);
				
				$callable = array($model[$key], $method);
	
				if (is_callable($callable)) {
					$output = call_user_func_array($callable, $args);
				} else {
					throw new \Exception('Error: Could not call model/' . $route . '!');
				}					
			}
			
			// Trigger the post events
			$result = $registry->get('event')->trigger('model/' . $trigger . '/after', array(&$route, &$args, &$output));
			
			if ($result && !$result instanceof Exception) {
				$output = $result;
			}
						
			return $output;
		};
	}	
}