<?php

class StringHelper
{

    static $POSTAL_CODE_REGEX = array(
        'nl' => '/^[1-9][0-9]{3} ?(?!sa|sd|ss)[a-z]{2}$/i',
        'be' => '/^[1-9][0-9]{3}$/i',
        'de' => '/^(?!01000|99999)(0[1-9]\d{3}|[1-9]\d{4})$/i',
        'at' => '/^[1-9]{1}[0-9]{3}$/i',
        'ch' => '/^[1-9]{1}[0-9]{3}$/i',
        'fr' => '/^(F-)?((2[A|B])|[0-9]{2})[0-9]{3}$/i',
    );

    /**
     * @param string $camelCased
     * @return string
     */
    public static function camelCasedToUnderScored($camelCased)
    {
        return strtolower(preg_replace('/(?<!^)[A-Z]/', '_$0', $camelCased));
    }

    /**
     * @param string $url
     * @param array $parameters
     * key as parameter name, value as parameter value. If value is null, only the parameter
     * is added without =value
     * @return string
     */
    public static function addParameters($url, $parameters)
    {
        foreach ($parameters as $key => $val) {
            if (strpos($url, $key) !== false) {
                continue;
            }
            $separator = strpos($url, '?') === false ? '?' : '&';
            $url .= $separator.$key;
            if ($val !== null) {
                $url .= '=' . $val;
            }
        }

        return $url;
    }

    public static function toEuro($cents, $decimals=2)
    {
        return number_format($cents / 100, $decimals, ',', '.');
    }

    /**
     * @param string $postalCode
     * @param string[] $limitToCountryCodes: country codes
     * @param string[]: country codes
     */
    public static function postalCodeToCountryCodes($postalCode, $limitToCountryCodes = null, $liberal = true)
    {
        $limitToCountryCodes = $limitToCountryCodes ?: array_keys(self::$POSTAL_CODE_REGEX);
        $countryCodes = [];
        foreach ($limitToCountryCodes as $countryCode) {
            if (self::isCountryCodePostalCode($countryCode, $postalCode,$liberal)) {
                $countryCodes[] = $countryCode;
            }
        }

        return $countryCodes;
    }

    /**
     * @param $countryCode
     * @param $postalCode
     * @param bool $liberal: removes all whitespace
     * @return bool
     */
    public static function isCountryCodePostalCode($countryCode, $postalCode, $liberal = true)
    {
        $countryCode = strtolower($countryCode);
        if ($liberal) {
            $postalCode = trim($postalCode);
            $postalCode = str_replace('  ', '', $postalCode);
            $postalCode = str_replace(' ', '', $postalCode);
        }

        if (!isset(self::$POSTAL_CODE_REGEX[$countryCode])) {
            return null;
        }

        $pattern = self::$POSTAL_CODE_REGEX[$countryCode];
        return preg_match($pattern, $postalCode) === 1;
    }

    public static function isDutchPostalCode($postalCode, $liberal = true)
    {
        return self::isCountryCodePostalCode('nl', $postalCode, $liberal);
    }

    public static function isBelgianPostalCode($postalCode, $liberal = true)
    {
        return self::isCountryCodePostalCode('be', $postalCode, $liberal);
    }

    public static function getPayproCovertCharacters()
    {
        return [
            '‘' => '\'',
//            'Ü' => 'Ue',
//            'ü' => 'ue',
//            'Ö' => 'Oe',
//            'ö' => 'oe',
//            'Ä' => 'Ae',
//            'ä' => 'ae',
//            'ß' => 'ss',
        ];
    }

    /**
     * @deprecated paypro has fixed the issue
     * @param string $s
     * @return string
     */
    public static function formInputToPayproCharacters($s)
    {
        foreach (self::getPayproCovertCharacters() as $from => $to) {
            $s = str_replace($from, $to, $s);
        }
        $s = htmlentities($s, ENT_NOQUOTES);

        return $s;
    }

    /**
     * Paypro has fixed this most of this issue, however this method must be kept here
     * for some sale and xml properties contain html entities.
     * It seems that it is still relevant for all sales that contain an ' character, that becomes &#39; in the xml
     * @param string $s
     * @return string
     */
    public static function payproCharactersToXml($s)
    {
        $s = str_replace('&amp;', '&', $s);
        $s = html_entity_decode($s, ENT_NOQUOTES);

        return $s;
    }
}