<?php

use Senh\Lib\Managers\InternalProductManager;
use Senh\Lib\Models\InternalProductPropertiesModel;

/**
 * Als een iemand vanuit Duitsland via internet een fysiek boek koopt van S&H, dan is dit een belastbare prestatie in
 * Nederland. De levering is belast met 6% Nederlandse BTW. Het maakt verder niet uit of dit een Nederlandstalig of
 * Duitstalig boek is. Het Duitse tarief van 7% is alleen van toepassing als er vanuit Duitsland geleverd wordt,
 * dus bijvoorbeeld als S&H in Duitsland een magazijn heeft. Verleggen is inderdaad niet mogelijk aan particulieren
 * zonder BTW nummer.
 *
 * BELANGRIJK: Hieruit volgt dat country altijd 'nl' moet zijn totdat S&H fysiek internationaal gaat.
 */
class VatHelper
{
    const VAT_LOW_OLD = 6; // Netherlands
    const VAT_LOW = 9; // Netherlands
    const VAT_HIGH = 21; // Netherlands
    const VAT_HIGH_BE = 21; // Belguim
    const VAT_LOW_DE = 7; // Germany
    const VAT_HIGH_DE = 19; // Germany
    const VAT_HIGH_FR = 20; // France
    const VAT_HIGH_AT = 20; // Austria
    const VAT_HIGH_CH = 8; // Switserland

    /**
     * @param int $vat
     * @param string $country
     * @param DateTime $date
     * @return bool
     * @throws Exception
     */
    public static function isVatLow($vat, $date = null, $country = 'nl')
    {
        return $vat === self::getVatLowPercentage($date, $country);
    }

    /**
     * @param int $vat
     * @return bool
     */
    public static function isVatHigh($vat, $date = null, $country = 'nl')
    {
        return $vat === self::getVatHighPercentage($date, $country);
    }

    /**
     * @param DateTime $date
     * @param string $country
     * @return float
     * @throws Exception
     */
    public static function getVatHighPercentage($date = null, $country = 'nl')
    {
        $country = strtolower($country);
        $date = $date ?: new DateTime('now', new DateTimeZone('Europe/Amsterdam'));
        if ($date->getTimezone()->getName() !== 'Europe/Amsterdam')
        {
            $date = clone $date;
            $date->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        }

        switch ($country) {
            case 'nl':
                return self::VAT_HIGH;
            case 'be':
                return self::VAT_HIGH_BE;
            case 'de':
                return self::VAT_HIGH_DE;
            case 'fr':
                return self::VAT_HIGH_FR;
            case 'at':
                return self::VAT_HIGH_AT;
            case 'ch':
                return self::VAT_HIGH_CH;
            default:
                throw new Exception("Unknown vat for country: $country");
        }
    }

    /**
     * @param DateTime $date
     * @param string $country
     * @return float
     * @throws Exception
     */
    public static function getVatLowPercentage($date = null, $country = 'nl')
    {
        $country = strtolower($country);
        $date = $date ?: new DateTime('now', new DateTimeZone('Europe/Amsterdam'));
        if ($date->getTimezone()->getName() !== 'Europe/Amsterdam')
        {
            $date = clone $date;
            $date->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        }

        switch ($country) {
            case 'nl':
                if ($date->format('Y-m-d') < '2019-01-01') {
                    return self::VAT_LOW_OLD;
                }
                return self::VAT_LOW;
            case 'de':
                return self::VAT_LOW_DE;
            default:
                throw new Exception("Unknown vat for country: $country");
        }
    }

    /**
     * @param $productCategory
     * @param $productType
     * @param DateTime|null $date
     * @param string $country
     * @throws Exception
     * @return float
     */
    public static function getVatPercentageByCategoryAndType($productCategory, $productType, $date = null, $country = 'nl')
    {
        $country = strtolower($country);
        $date = $date ?: new DateTime('now', new DateTimeZone('Europe/Amsterdam'));
        if ($date->getTimezone()->getName() !== 'Europe/Amsterdam')
        {
            $date = clone $date;
            $date->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        }

        // before 2019, only 'nl' vat rates were used (meaning that MOSS countries' vat rates were set to the Dutch VAT_HIGH)
        if ($date->format('Y-m-d') < '2019-01-01') {
            $country = 'nl';
        }

        // if not MOSS product or not MOSS country force country to 'nl'
        $country =
            self::isMossProduct($productCategory, $productType) &&
            self::isMossCountry($country, $date)
                ? $country
                : 'nl';

        // physical products
        if ($productType === InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL) {
            return self::getVatLowPercentage($date, $country);
        }

        // digital books
        if (
            $productCategory === InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK &&
            $productType === InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL
        ) {
            switch ($country) {
                case 'de':
                    return self::getVatLowPercentage($date, $country);
                default:
                    return self::getVatHighPercentage($date, $country);
            }

        }

        // remaining products (e.g. digital program 'het ultieme afslank programma')
        return self::getVatHighPercentage($date, $country);
    }

    /**
     * @param string $productCategory
     * @param string $productType
     * @return bool
     */
    public static function isMossProduct($productCategory, $productType)
    {
        return $productType === InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL;
    }

    /**
     * @param string $country
     * @param \DateTime|null $date: if null takes current date
     * @return bool
     * @throws Exception
     */
    public static function isMossCountry($country, $date = null)
    {
        $availableMossCountries = self::getAvailableMossCountries($date);

        return in_array($country, $availableMossCountries);
    }

    /**
     * Get available moss countries for s&h based given a (transaction) date
     * @param \DateTime|null $date: if null takes current date
     * @return string[]
     * @throws Exception
     */
    public static function getAvailableMossCountries($date = null)
    {
        $date = $date ?: new DateTime('now', new DateTimeZone('Europe/Amsterdam'));
        if ($date->getTimezone()->getName() !== 'Europe/Amsterdam')
        {
            $date = clone $date;
            $date->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        }

        $availableMossCountries = ['nl', 'be'];
        if ($date->format('Y-m-d') >= '2018-07-01') {
            $availableMossCountries[] = 'de';
        }
        if ($date->format('Y-m-d') >= '2018-09-01') {
            $availableMossCountries[] = 'fr';
        }
        if ($date->format('Y-m-d') >= '2019-01-01') {
            $availableMossCountries[] = 'ch';
            $availableMossCountries[] = 'at';
        }

        return $availableMossCountries;
    }

    /**
     * @param null $date
     * @param string $country
     * @return float
     * @throws Exception
     */
    public static function getVatBook($date = null, $country = 'nl')
    {
        return self::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL,
            $date,
            $country
        );
    }

    /**
     * @param null $date
     * @param string $country
     * @return float
     * @throws Exception
     */
    public static function getVatEBook($date = null, $country = 'nl')
    {
        return self::getVatPercentageByCategoryAndType(
            InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK,
            InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL,
            $date,
            $country
        );
    }

    /**
     * @param SaleModel $sale
     * @return string|null
     */
    public static function induceSaleConsumerCountryCode(SaleModel $sale)
    {
        if ($sale->getCountryCode()) {
            return $sale->getCountryCode();
        }

        if ($sale->getIbanCountry()) {
            return $sale->getIbanCountry();
        }

        $internalProductModel = InternalProductManager::getInstance()->getSingle($sale);
        if ($internalProductModel) {
            return $internalProductModel->getProductCountryCode();
        }

        return 'nl';
    }

    /**
     * @param SequenceModel $sequence
     * @return int
     * @throws Exception
     */
    public static function induceSequenceVat(SequenceModel $sequence)
    {
        if (!$sequence->getSale()) {
            throw new Exception('Sale must be set.');
        }

        if (!$sequence->getFirstTransaction()) {
            throw new Exception('The transactions must be set.');
        }

        $internalProductManager = InternalProductManager::getInstance();
        $productId = $sequence->getSale()->getProductId();
        $isFysiek1 = $internalProductManager->isPhysical1($productId);
        $internalProduct = $internalProductManager->getSingle($productId);
        $sequenceDateUtc = $sequence->getDate();
        $vatDateNl = clone $sequence->getFirstTransaction()->getDate();
        $vatDateNl->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        $saleDateNl = clone $sequence->getSale()->getDate();
        $saleDateNl->setTimezone(new DateTimeZone('Europe/Amsterdam'));
        $consumerCountryCode = self::induceSaleConsumerCountryCode($sequence->getSale());

        if ($vatDateNl->format('Y-m-d') >= '2019-01-01') {
            // from 2019, we simply define product type and category to get the vat rate
            $productCategory = null;
            $productType = null;
            if ($internalProduct) {
                $productCategory = $sequence->getSequenceNumber() === 1 ? $internalProduct->getProductCategory1() : $internalProduct->getProductCategoryN();
                $productType = $sequence->getSequenceNumber() === 1 ? $internalProduct->getProductType1() : $internalProduct->getProductTypeN();
            }

            if (!$productCategory || !$productType) {
                // defaults
                if ($sequence->getSequenceNumber() === 1) {
                    $productCategory = InternalProductPropertiesModel::PRODUCT_CATEGORY_BOOK;
                    $productType = $isFysiek1
                        ? InternalProductPropertiesModel::PRODUCT_TYPE_PHYSICAL
                        : InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL;
                } else {
                    $productCategory = InternalProductPropertiesModel::PRODUCT_CATEGORY_PROGRAM;
                    $productType = InternalProductPropertiesModel::PRODUCT_TYPE_DIGITAL;
                }
            }

            return  VatHelper::getVatPercentageByCategoryAndType(
                $productCategory,
                $productType,
                $vatDateNl,
                $consumerCountryCode
            );
        }

        // Before 2019 there were many bugs + only dutch vat rates were used. We keep this isolated.
        $consumerCountryCode = 'nl';
        if (!$isFysiek1) {
            // digital1 products were never vat corrected, so we can handle those separately in this block
            if ($sequence->getSequenceNumber() === 1) {
                return self::getVatHighPercentage($sequenceDateUtc);
            }

            // from here: both digitalN and physicalN
            if (!$internalProduct) {
                // probably HUAP (digital program), for other recurring products always have a internalProductModel defined
                return self::getVatHighPercentage($sequenceDateUtc);
            }

            $productCategory = $internalProduct->getProductCategoryN();
            $productType = $internalProduct->getProductTypeN();
            if (!$productCategory || !$productType) {
                // little fallback just in case, HUAP is most likely
                return self::getVatHighPercentage($sequenceDateUtc);
            }

            return  VatHelper::getVatPercentageByCategoryAndType(
                $productCategory,
                $productType,
                $vatDateNl,
                $consumerCountryCode
            );
        }


        // from here: only product ids which match physical1

        // bij alle niet interne producten of alle interne producten zonder digitaal programma is er nooit iets fout gegaan dus vat laag bij sequentie 1 en hoog bij sequentie > 1
        if (!$internalProduct || !$internalProduct->hasPropertiesN()) {
            return $sequence->getSequenceNumber() === 1 ? self::getVatLowPercentage($vatDateNl) : self::getVatHighPercentage($vatDateNl);
        }

        // alle fysieke producten met programma waren 21% voor de datum 2017-04-11 07:19
        if (DateHelper::toSqlFormat($sequenceDateUtc) < '2017-04-11 07:19') {
            return self::getVatHighPercentage($sequenceDateUtc);
        }

        // bug: tussen 2017-11-14 13:16 en 2018-02-27 17:30 is een sequence > 1 niet omgezet naar 21% bij nieuwe sales!
        if (
            // vat error (see commit: "import intermediate" where vat_high constant was removed from SaleModel)), but was merged to master on 2017-11-14 13:16 (merge branch 'import')
            DateHelper::toSqlFormat($sequence->getSale()->getDate()) > '2017-11-14 13:16' &&
            // correction of vat error
            DateHelper::toSqlFormat($sequenceDateUtc) < '2018-02-27 16:30'
        ) {
            return self::getVatLowPercentage($vatDateNl);
        }

        // de resterende fysieke producten met abonnement
        $productCategory = $sequence->getSequenceNumber() === 1 ? $internalProduct->getProductCategory1() : $internalProduct->getProductCategoryN();
        $productType = $sequence->getSequenceNumber() === 1 ? $internalProduct->getProductType1() : $internalProduct->getProductTypeN();
        if (!$productCategory || !$productType) {
            // little fallback just in case
            return $sequence->getSequenceNumber() === 1 ? self::getVatLowPercentage($vatDateNl) : self::getVatHighPercentage($vatDateNl);
        }

        return self::getVatPercentageByCategoryAndType(
            $productCategory,
            $productType,
            $vatDateNl,
            $consumerCountryCode
        );
    }

    public static function calcVatPart($fullPrice, $vatRate)
    {
        return $fullPrice - static::calcNetPrice($fullPrice, $vatRate);
    }

    public static function calcNetPrice($fullPrice, $vatRate)
    {
        return $fullPrice / (1 + $vatRate / 100);
    }

    public static function calcVat($netPrice, $vatRate)
    {
        return $netPrice * ($vatRate / 100);
    }

    public static function calcFullPrice($netPrice, $vatRate)
    {
        return $netPrice * (1 + $vatRate / 100);
    }
}