<?php

namespace Senh\Lib\Helpers;

use Doctrine\Instantiator\Exception\InvalidArgumentException;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

use StringHelper;

class StringHelperTest extends TestCase
{
    public static $VALID_POSTAL_CODES = [
        'nl' => '1234AB',
        'be' => '1234',
        'de' => '12345',
        'at' => '1234',
        'ch' => '1234',
        'fr' => '12345',
    ];

    public static $INVALID_POSTAL_CODES = [
        'nl' => '1234ABX',
        'be' => '1234X',
        'de' => '12345X',
        'at' => '1234X',
        'ch' => '1234X',
        'fr' => '12345X',
    ];

    public function testValidPostalCodes()
    {
        foreach (self::$VALID_POSTAL_CODES as $countryCode => $postalCode) {
            $actual = StringHelper::isCountryCodePostalCode($countryCode, $postalCode);
            $this->assertTrue($actual, 'Country Code ' . $countryCode);
        }

        $this->assertTrue(StringHelper::isDutchPostalCode(self::$VALID_POSTAL_CODES['nl']));
        $this->assertTrue(StringHelper::isBelgianPostalCode(self::$VALID_POSTAL_CODES['be']));
    }

    public function testInvalidPostalCodes()
    {
        foreach (self::$INVALID_POSTAL_CODES as $countryCode => $postalCode) {
            $actual = StringHelper::isCountryCodePostalCode($countryCode, $postalCode);
            $this->assertFalse($actual, 'Country Code ' . $countryCode);
        }
    }

    public function testPostalCodeToCountryCodes()
    {
        $actual = StringHelper::postalCodeToCountryCodes('1234', ['be', 'at']);
        $expected = ['be', 'at'];
        $this->assertEquals($expected, $actual);
    }

    public function testFormInputToPayproCharacters()
    {
        $expected = '"';
        $actual = StringHelper::formInputToPayproCharacters('"');
        $this->assertEquals($expected, $actual);

        $expected = '\'';
        $actual = StringHelper::formInputToPayproCharacters('\'');
        $this->assertEquals($expected, $actual);

//        $expected = 'ue';
//        $actual = StringHelper::formInputToPayproCharacters('ü');
//        $this->assertEquals($expected, $actual);
//
//        $expected = 'Ue';
//        $actual = StringHelper::formInputToPayproCharacters('Ü');
//        $this->assertEquals($expected, $actual);
//
//        $input = 'ß';
//        $expected = 'ss';
//        $actual = StringHelper::formInputToPayproCharacters($input);
//        $this->assertEquals($expected, $actual);

        $expected = '&eacute;';
        $actual = StringHelper::formInputToPayproCharacters('é');
        $this->assertEquals($expected, $actual);

        $expected = array_values(StringHelper::getPayproCovertCharacters());
        $actual = [];
        foreach (StringHelper::getPayproCovertCharacters() as $from => $to) {
            $actual[] = StringHelper::formInputToPayproCharacters($from);
        }
        $this->assertEquals($expected, $actual);
    }

    public function testPayproCharactersToXml()
    {
        $expected = '"'; // unchanged
        $actual = StringHelper::payproCharactersToXml('"');
        $this->assertEquals($expected, $actual);

        $expected = '\''; // unchanged
        $actual = StringHelper::payproCharactersToXml('\'');
        $this->assertEquals($expected, $actual);

        $expected = 'ue';// unchanged
        $actual = StringHelper::payproCharactersToXml('ue');
        $this->assertEquals($expected, $actual);

        $expected = 'Ue'; // unchanged
        $actual = StringHelper::payproCharactersToXml('Ue');
        $this->assertEquals($expected, $actual);

        $expected = '&';
        $actual = StringHelper::payproCharactersToXml('&amp;');
        $this->assertEquals($expected, $actual);

        $expected = 'é';
        $actual = StringHelper::payproCharactersToXml('&amp;eacute;');
        $this->assertEquals($expected, $actual);

        $expected = 'é';
        $actual = StringHelper::payproCharactersToXml('&eacute;');
        $this->assertEquals($expected, $actual);

        $expected = array_values(StringHelper::getPayproCovertCharacters()); // unchanged
        $actual = [];
        foreach (StringHelper::getPayproCovertCharacters() as $to => $from) {
            $actual[] = StringHelper::payproCharactersToXml($from);
        }
        $this->assertEquals($expected, $actual);
    }

    public function testFromFormInputToPayproToXml()
    {
        $input    = 'Böeßnäl/Vîgtl ‘Nürd-Brébènt';
        $expected = 'Böeßnäl/Vîgtl \'Nürd-Brébènt';
        $t = StringHelper::formInputToPayproCharacters($input);
        $actual = StringHelper::payproCharactersToXml($t);
        $this->assertEquals($expected, $actual);
    }
}
