<?php

namespace WishListMember\Autoresponders;

if (! class_exists('\WLM_Automizy')) {
    require_once wishlistmember_instance()->plugin_dir . '/extlib/wlm-automizy.php';
}

class Automizy
{
    /**
     * Handle static method calls.
     *
     * This method is invoked when invoking inaccessible methods in a static context.
     * It checks if the API interface is available and calls the requested method
     * on the interface with the provided arguments.
     *
     * @param string $name The name of the method being called.
     * @param array  $args The arguments passed to the method.
     */
    public static function __callStatic($name, $args)
    {
        $interface = self::_interface();
        if ($interface->api()) {
            call_user_func_array([$interface, $name], $args);
        }
    }

    /**
     * Retrieves the singleton instance of the AutomizyInterface.
     *
     * This method ensures that only one instance of the AutomizyInterface
     * is created and reused throughout the class. It initializes the instance
     * on the first call and returns the same instance on subsequent calls.
     *
     * @return AutomizyInterface The singleton instance of the AutomizyInterface.
     */
    public static function _interface()
    {
        static $interface;
        if (! $interface) {
            $interface = new AutomizyInterface();
        }
        return $interface;
    }
}

class AutomizyInterface
{
    /**
     * Stores the settings for Automizy integration
     *
     * @var string
     */
    private $settings = '';
    /**
     * Stores the API key for Automizy
     *
     * @var string|null
     */
    private $api_key = null;

    /**
     * Initializes the Automizy API connection if the necessary classes are available
     */
    public function __construct()
    {
        $this->automizy_api = false;
        // Make sure that WLM active and infusiosnsoft connection is set.
        if (class_exists('\WLM_Automizy')) {
            $this->settings = wlm_or(( new \WishListMember\Autoresponder('automizy') )->settings, false);
            // Initilize automizy api connection.
            if ($this->settings && ! empty($this->settings['api_key'])) {
                $this->automizy_api = new \WLM_Automizy($this->settings['api_key']);
                $lists_ret          = $this->automizy_api->get('smart-lists');
                if (! $this->automizy_api->is_success()) {
                    $this->automizy_api = false;
                }
            }
        }
    }

    /**
     * Returns the Automizy API instance
     */
    public function api()
    {
        return $this->automizy_api;
    }

    /**
     * Processes tags for a user based on the specified action and levels
     *
     * @param  array|string $levels The levels to process tags for.
     * @param  string       $action The action to perform (add, cancel, rereg, remove).
     * @param  array        $data   User data including email and names.
     * @return array|boolean - Returns an array of errors or true on success
     */
    public function process_tags($levels, $action, $data)
    {
        if (! $this->automizy_api) {
            return [
                'errstr' => 'Unable to process tags. No API Connection.',
                'errno'  => 1,
            ];
        }
        $levels = (array) $levels;
        if (count($levels) <= 0) {
            return [
                'errstr' => 'No Levels Found',
                'errno'  => 1,
            ];// No levels, no need to continue.
        }
        if (! isset($data['user_email']) || empty($data['user_email'])) {
            return [
                'errstr' => 'Email address not found',
                'errno'  => 1,
            ];
        }
        if (! in_array($action, ['add', 'cancel', 'rereg', 'remove'], true)) {
            return [
                'errstr' => 'Invalid action',
                'errno'  => 1,
            ];
        }

        $errors = [];
        // Add the tags for each level.
        foreach ((array) $levels as $level) {
            $error = [];

            if ('wishlistmember_payperpost_added' === current_action() || 'wishlistmember_payperpost_removed' === current_action()) {
                $level = 'payperpost-' . $level;
            }

            $apply_tags = isset($this->settings[ $level ][ $action ]['apply_tag']) ? $this->settings[ $level ][ $action ]['apply_tag'] : false;
            $apply_tags = ! empty($apply_tags) ? $apply_tags : false;
            $remove_tag = isset($this->settings[ $level ][ $action ]['remove_tag']) ? $this->settings[ $level ][ $action ]['remove_tag'] : false;
            $remove_tag = ! empty($remove_tag) ? $remove_tag : false;

            $list_add    = isset($this->settings[ $level ][ $action ]['list_add']) ? $this->settings[ $level ][ $action ]['list_add'] : false;
            $list_add    = ! empty($list_add) ? $list_add : false;
            $list_remove = isset($this->settings[ $level ][ $action ]['list_remove']) ? $this->settings[ $level ][ $action ]['list_remove'] : false;
            $list_remove = ! empty($list_remove) ? $list_remove : false;

            $c = $this->automizy_api->get("contacts/{$data['user_email']}");
            if ($c) {
                $tags = array_merge($c['tags'], $tags);
                $tags = array_unique($tags);

                $new_lists = [];
                foreach ($c['smartLists'] as $key => $value) {
                    $new_lists[] = $value['id'];
                }
                // Lets remove first.
                if ($list_remove) {
                    $new_lists = array_diff($new_lists, $list_remove);
                }
                if ($list_add) {
                    $new_lists[] = $list_add;
                }
                $new_lists = array_unique($new_lists);

                $params = [
                    'customFields' => [
                        'firstname' => $data['first_name'],
                        'lastname'  => $data['last_name'],
                    ],
                    'smartLists'   => $new_lists,
                ];
                if ($remove_tag) {
                    $params['removeTags'] = $remove_tag;
                }
                if ($apply_tags) {
                    $params['addTags'] = $apply_tags;
                }

                $c = $this->automizy_api->patch("contacts/{$data['user_email']}", $params);
            } else {
                $params   = [
                    'email'        => $data['user_email'],
                    'customFields' => [
                        'firstname' => $data['first_name'],
                        'lastname'  => $data['last_name'],
                    ],
                    'tags'         => $apply_tags,
                ];
                $list_add = $list_add ? $list_add : 1;
                $c        = $this->automizy_api->post("smart-lists/{$list_add}/contacts", $params);
            }

            if (! $this->automizy_api->is_success() && ! empty($this->automizy_api->get_last_error())) {
                $errors[ $level ] = $this->automizy_api->get_last_error();
            }
        }
        return count($errors) ? wlm_maybe_serialize($errors) : true; // Success.
    }

    /**
     * Adds a data item to the processing queue
     *
     * @param array   $data    The data to queue.
     * @param boolean $process Whether to process the queue immediately.
     */
    public function add_queue($data, $process = true)
    {
        $wishlist_api_queue_instance = new \WishListMember\API_Queue();
        $qname                       = 'automizyar' . time();
        $data                        = wlm_maybe_serialize($data);
        $wishlist_api_queue_instance->add_queue($qname, $data, 'For Queueing');
        if ($process) {
            $this->process_queue();
        }
    }

    /**
     * Processes items in the queue
     *
     * @param integer $recnum Number of records to process.
     * @param integer $tries  Number of attempts for each record.
     */
    public function process_queue($recnum = 10, $tries = 3)
    {
        if (! $this->automizy_api) {
            return;
        }
        $wishlist_api_queue_instance = new \WishListMember\API_Queue();
        $last_process                = get_option('WLM_AUTORESPONDER_AUTOMIZY_LastProcess');
        $current_time                = time();
        $tries                       = $tries > 1 ? (int) $tries : 3;
        $error                       = false;
        // Lets process every 10 seconds.
        if (! $last_process || ( $current_time - $last_process ) > 10) {
            $queues = $wishlist_api_queue_instance->get_queue('automizyar', $recnum, $tries, 'tries,name');
            foreach ($queues as $queue) {
                $data = wlm_maybe_unserialize($queue->value);
                if ('new' === $data['action']) {
                    $res = $this->new_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('add' === $data['action']) {
                    $res = $this->add_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('remove' === $data['action']) {
                    $res = $this->remove_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('cancel' === $data['action']) {
                    $res = $this->cancel_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('rereg' === $data['action']) {
                    $res = $this->rereg_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                } elseif ('delete' === $data['action']) {
                    $res = $this->delete_user_tags_hook($data['uid'], $data['levels'], $data['data']);
                }

                if (true !== $res) {
                    $d = [
                        'notes' => $res,
                        'tries' => $queue->tries + 1,
                    ];
                    $wishlist_api_queue_instance->update_queue($queue->ID, $d);
                    $error = true;
                } else {
                    $wishlist_api_queue_instance->delete_queue($queue->ID);
                    $error = false;
                }
            }
            // Save the last processing time.
            if ($error) {
                $current_time = time();
                if ($last_process) {
                    update_option('WLM_AUTORESPONDER_AUTOMIZY_LastProcess', $current_time);
                } else {
                    add_option('WLM_AUTORESPONDER_AUTOMIZY_LastProcess', $current_time);
                }
            }
        }
    }

    /**
     * Queues a new user for tag processing
     *
     * @param integer|null $uid   User ID.
     * @param array|null   $udata User data.
     */
    public function new_user_tags_hook_queue($uid = null, $udata = null)
    {
        // Part of the Fix for issue where Add To levels aren't being processed.
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        // Don't add the data into the queue if it's from a temp account.
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'new',
            'levels' => (array) $udata['wpm_id'],
            'data'   => $udata,
        ];
        $this->add_queue($data);
    }

    /**
     * Processes new user tags
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function new_user_tags_hook($uid, $levels, $data)
    {
        $tempacct = 'temp_' . md5($data['orig_email']) === $data['email'];
        if ($tempacct) {
            return; // If temp account used by sc, do not process.
        }
        return $this->process_tags($levels, 'add', $data);
    }

    /**
     * Queues a user for adding tags
     *
     * @param integer $uid       User ID.
     * @param string  $addlevels Levels to add.
     */
    public function add_user_tags_hook_queue($uid, $addlevels = '')
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }

        $udata               = [];
        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'add',
            'levels' => $addlevels,
            'data'   => $udata,
        ];
        // Fix for issue where Add To levels aren't being processed.
        // If the data is from a temp account then add it to the queue API and don't process it for now.
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            $this->add_queue($data, 0);
        } elseif (isset(wlm_post_data()['SendMail'])) {
            // This elseif condition fixes the issue where members who are added via.
            // WLM API aren't being processed by the Integration.
            $this->add_queue($data, 0);
        } else {
            $this->add_queue($data);
        }
    }

    /**
     * Processes adding tags for a user
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function add_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        // Make sure that info are updated.
        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'add', $data);
    }

    /**
     * Queues a user for removing tags
     *
     * @param integer $uid           User ID.
     * @param string  $removedlevels Levels to remove.
     */
    public function remove_user_tags_hook_queue($uid, $removedlevels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $removedlevels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'remove',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes removing tags for a user
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function remove_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'remove', $data);
    }

    /**
     * Queues a user for canceling tags
     *
     * @param integer $uid          User ID.
     * @param string  $cancellevels Levels to cancel.
     */
    public function cancel_user_tags_hook_queue($uid, $cancellevels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $cancellevels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'cancel',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes canceling tags for a user
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function cancel_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'cancel', $data);
    }

    /**
     * Queues a user for re-registering tags
     *
     * @param integer $uid    User ID.
     * @param string  $levels Levels to re-register.
     */
    public function rereg_user_tags_hook_queue($uid, $levels = '')
    {
        // Lets check for PPPosts.
        $levels = (array) $levels;
        foreach ($levels as $key => $level) {
            if (false !== strrpos($level, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (count($levels) <= 0) {
            return;
        }

        $data = [
            'uid'    => $uid,
            'action' => 'rereg',
            'levels' => $levels,
            'data'   => [],
        ];
        $this->add_queue($data);
    }

    /**
     * Processes re-registering tags for a user
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function rereg_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $data['first_name'] = $user->first_name;
        $data['last_name']  = $user->last_name;
        $data['user_email'] = $user->user_email;
        $data['username']   = $user->user_login;
        $levels             = (array) $levels;
        return $this->process_tags($levels, 'rereg', $data);
    }

    /**
     * Queues a user for deletion
     *
     * @param integer $uid User ID.
     */
    public function delete_user_hook_queue($uid)
    {
        if (! $this->automizy_api) {
            return;
        }

        $levels = wishlistmember_instance()->get_membership_levels($uid);
        foreach ($levels as $key => $lvl) {
            if (false !== strpos($lvl, 'U-')) {
                unset($levels[ $key ]);
            }
        }
        if (! is_array($levels) || count($levels) <= 0) {
            return; // Lets return if no level was found.
        }

        $user = get_userdata($uid);
        if (! $user) {
            return;
        }

        $udata               = [];
        $udata['first_name'] = $user->first_name;
        $udata['last_name']  = $user->last_name;
        $udata['user_email'] = $user->user_email;
        $udata['username']   = $user->user_login;
        $data                = [
            'uid'    => $uid,
            'action' => 'delete',
            'levels' => $levels,
            'data'   => $udata,
        ];
        $this->add_queue($data);
        return;
    }

    /**
     * Processes deletion of user tags
     *
     * @param integer $uid    User ID.
     * @param array   $levels Membership levels.
     * @param array   $data   User data.
     */
    public function delete_user_tags_hook($uid, $levels, $data)
    {
        $user = get_userdata($uid);
        if (! $user) {
            return;
        }
        if (false !== strpos($user->user_email, 'temp_') && 37 === strlen($user->user_email) && false === strpos($user->user_email, '@')) {
            return;
        }

        $levels = (array) $levels;
        return $this->process_tags($levels, 'remove', $data);
    }
}
