<?php
/**
 * @license MIT
 *
 * Modified by caseproof on 06-January-2025 using Strauss.
 * @see https://github.com/BrianHenryIE/strauss
 */

declare(strict_types=1);

namespace WishListMember\Http\Adapter\Guzzle7;

use WishListMember\GuzzleHttp\Client as GuzzleClient;
use WishListMember\GuzzleHttp\ClientInterface;
use WishListMember\GuzzleHttp\HandlerStack;
use WishListMember\GuzzleHttp\Middleware;
use WishListMember\GuzzleHttp\Utils;
use WishListMember\Http\Client\HttpAsyncClient;
use WishListMember\Http\Client\HttpClient;
use WishListMember\Psr\Http\Message\RequestInterface;
use WishListMember\Psr\Http\Message\ResponseInterface;

/**
 * HTTP Adapter for Guzzle 7.
 *
 * @author Tobias Nyholm <tobias.nyholm@gmail.com>
 */
final class Client implements HttpClient, HttpAsyncClient
{
    /**
     * @var ClientInterface
     */
    private $guzzle;

    public function __construct(?ClientInterface $guzzle = null)
    {
        if (!$guzzle) {
            $guzzle = self::buildClient();
        }

        $this->guzzle = $guzzle;
    }

    /**
     * Factory method to create the Guzzle 7 adapter with custom Guzzle configuration.
     */
    public static function createWithConfig(array $config): Client
    {
        return new self(self::buildClient($config));
    }

    /**
     * {@inheritdoc}
     */
    public function sendRequest(RequestInterface $request): ResponseInterface
    {
        return $this->sendAsyncRequest($request)->wait();
    }

    /**
     * {@inheritdoc}
     */
    public function sendAsyncRequest(RequestInterface $request)
    {
        $promise = $this->guzzle->sendAsync($request);

        return new Promise($promise, $request);
    }

    /**
     * Build the Guzzle client instance.
     */
    private static function buildClient(array $config = []): GuzzleClient
    {
        $handlerStack = new HandlerStack(Utils::chooseHandler());
        $handlerStack->push(Middleware::prepareBody(), 'prepare_body');
        $config = array_merge(['handler' => $handlerStack], $config);

        return new GuzzleClient($config);
    }
}
